# run_skeleton_task.py (Executable Code Fully Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration to identify missing predicates (e.g., (drawer-unlocked drawer1)).'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # -------------------------------------------------------------
        # Exploration Phase: Identify if "drawer-unlocked" predicate is present
        # -------------------------------------------------------------
        # Suppose we need to manipulate drawer1
        # Try to execute 'execute_pull' on the drawer and handle.
        # If 'drawer-unlocked' is missing, execute exploration actions or display it.

        drawer_name = 'drawer1'
        # Many simulation environments call the handle something like 'drawer1_handle' or 'handle1'
        possible_handles = [k for k in positions.keys() if 'handle' in k and drawer_name in k]
        if possible_handles:
            handle_name = possible_handles[0]
        else:
            # fallback: any handle
            handle_candidates = [k for k in positions.keys() if 'handle' in k]
            handle_name = handle_candidates[0] if handle_candidates else None
        if not handle_name:
            print("[Exploration] No handle found for", drawer_name)
            return

        # The location might be supplied as well. Let's try positions['drawer1'], but fallback to a generic position otherwise
        if drawer_name in positions:
            drawer_pos = positions[drawer_name]
        else:
            drawer_pos = None
        if handle_name in positions:
            handle_pos = positions[handle_name]
        else:
            handle_pos = None

        # Find some location near drawer1/handle1 for the robot to move to
        robot_location_candidates = [k for k in positions.keys() if 'pose' in k or 'ready' in k or 'start' in k]
        if robot_location_candidates:
            robot_init_loc = robot_location_candidates[0]
            robot_init_pos = positions[robot_init_loc]
        else:
            robot_init_loc = None
            robot_init_pos = None

        # Step 1: Move robot to initial location if required (execute_go)
        if robot_init_loc and 'ready' not in robot_init_loc:
            try:
                print(f"[Exploration] Moving robot to starting location: {robot_init_loc}")
                obs, reward, done, info = execute_go(env, task, from_location='ready-pose', to_location=robot_init_loc)
            except Exception as e:
                print(f"[Exploration] Could not execute initial go: {e}")

        # Step 2: Try to pick the handle (execute_pick)
        # "execute_pick" may require additional information, but we assume handle is on floor or reachable
        try:
            print(f"[Exploration] Attempting to pick up handle {handle_name} at {robot_init_loc or 'drawer position'}")
            obs, reward, done, info = execute_pick(env, task, obj=handle_name, location=drawer_pos)
            print(f"[Exploration] Picked up handle {handle_name}")
        except Exception as e:
            print(f"[Exploration] Failed to pick handle: {e}")

        # Step 3: Try to pull the drawer (execute_pull), triggering failure if drawer is locked
        try:
            print(f"[Exploration] Attempting to pull {drawer_name} using handle {handle_name}")
            obs, reward, done, info = execute_pull(env, task, drawer=drawer_name, handle=handle_name, location=drawer_pos)
            print("[Exploration] Successfully pulled the drawer. It was unlocked.")
        except Exception as e:
            print(f"[Exploration] Failed to pull drawer (drawer may be locked): {e}")
            # We discovered missing predicate: (drawer-unlocked drawer1)
            print("[Exploration Result] Missing predicate detected: (drawer-unlocked drawer1)")

            # Normally, exploration code would now perform further steps, such as
            # attempting to unlock the drawer (if any unlock skill exists), or
            # trying alternate exploration actions.
            # But with only the predefined skills available (see below), no such unlock skill is provided.

        # Optional: Proceed with rest of plan if drawer-unlocked (here, no further plan is given since we only aim to discover the missing predicate).
        # In a complete task, this is where we could continue with e.g. pick-object, place-object, etc.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()