# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use skill functions directly as imported

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE: Identify Missing Predicate 'robot-free' ===
        # The feedback indicates the missing predicate is (robot-free).
        # Here, we attempt skill executions that would fail if 'robot-free' is not satisfied
        # and observe task feedback/exceptions.

        print("[Exploration] Testing skill preconditions related to 'robot-free'...")

        # Try to execute a skill that requires robot to be free
        exploration_done = False
        robot_free_status = None

        # We'll try to execute a 'execute_go' between two locations (if any exist)
        # and watch for exceptions or errors due to missing 'robot-free' predicate.

        locations = [k for k, v in positions.items() if v.get('type', '') == 'location']
        # If location info is not typed, fallback to keys containing 'location'
        if len(locations) < 2:
            locations = [k for k in positions if 'location' in k or 'pose' in k or 'place' in k]
        if len(locations) < 2:
            # Use any two distinct values as fallback
            locations = list(set(list(positions.keys())))[:2]

        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print(f"[Exploration] Attempting execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                exploration_done = True
                robot_free_status = True
                print("[Exploration] execute_go succeeded: robot-free probably holds.")
            except Exception as e:
                exploration_done = True
                robot_free_status = False
                print(f"[Exploration] execute_go failed with exception: {e}")
        else:
            print("[Exploration] Not enough location information for move test.")

        # Based on this, the exploration phase has attempted to determine if 'robot-free' is needed.

        if robot_free_status is False:
            print("[Exploration][Result] The 'robot-free' predicate may be missing or violated.")
            # Further actions could attempt other skills to elicit the predicate precondition.
        else:
            print("[Exploration][Result] The 'robot-free' predicate appears present or satisfied.")

        # === END OF EXPLORATION PHASE ===

        # === (Placeholder) Plan Execution Phase ===
        # In real tasks, this is where you would implement the oracle plan using available skills.
        # For now, after exploring the missing predicate, we can print overall results.

        print("===== Exploration Completed. Proceed to plan execution as required. =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
