# run_skeleton_task.py (Completed: Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- EXPLORATION PHASE: Find the Missing Predicate ---

        # Directly tied to domain knowledge and provided exploration predicates/skills, 
        # the goal is to determine which knowledge/predicate is missing in the original planning domain via interaction.

        # Using available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # As skill implementations (such as execute_pull, execute_pick, etc.) are already provided,
        # we now perform an exploration sequence to try to acquire missing knowledge for all objects.

        print("[Exploration] Starting exploration phase to determine unknown/missing predicates (like lock-known, weight-known, etc.)")

        # Example procedure: Try to pick and pull each object in the scene, and observe result/feedback/errors.

        # Get all objects and drawers
        all_objects = [k for k in positions.keys() if not k.startswith('robot') and not k.startswith('drawer')]
        all_drawers = [k for k in positions.keys() if k.startswith('drawer')]
        all_locations = list(set([v['location'] for v in positions.values() if 'location' in v]))
        robot_pose = positions.get('robot', {}).get('location', None)

        # Helper states - to be robust to missing information
        if robot_pose is None:
            # Try to get fallback; else, raise error
            robot_pose = all_locations[0] if all_locations else "ready-pose"
        
        # Tracking which predicates are known/detected
        check_results = {}

        # 1. Explore each object with pick and pull (check for lock-known and weight-known, etc.)
        for obj_name in all_objects:
            obj_info = positions[obj_name]
            obj_pos = obj_info.get('position', None)
            obj_loc = obj_info.get('location', None)

            if obj_loc is None and obj_pos is not None:
                obj_loc = obj_pos  # fallback

            result = {'weight_known': False, 'durability_known': False, 'lock_known': False, 'identified': False, 'temperature_known': False}
            
            # (a) Try to move to object location
            try:
                print(f"[Exploration] Moving robot to the location of '{obj_name}'...")
                obs, reward, done = execute_go(env, task, robot_pose, obj_loc)
                robot_pose = obj_loc # update pose
            except Exception as e:
                print(f"[Exploration][Warning] Failed to move to '{obj_name}' at '{obj_loc}': {e}")

            # (b) Try to pick (may trigger weight-known or durability-known depending on the skill mapping)
            try:
                print(f"[Exploration] Attempting to pick '{obj_name}' to check for knowledge...")
                obs, reward, done = execute_pick(env, task, obj_name, obj_loc)
                result['weight_known'] = True  # It's possible we know weight by picking
                print(f"[Exploration] Picked '{obj_name}'.")
            except Exception as e:
                print(f"[Exploration][Info] Could not pick '{obj_name}': {e}")
            
            # (c) Try to pull (may trigger lock-known predicate)
            # For pulling, object must be a handle or attached to a drawer, etc.
            try:
                # Guessing: check if it's a drawer handle, otherwise skip
                if obj_info.get('is_handle', False) or obj_name.endswith('_handle'):
                    for drawer_name in all_drawers:
                        print(f"[Exploration] Try to pull (execute_pull) with '{obj_name}' on '{drawer_name}' at '{obj_loc}'...")
                        obs, reward, done = execute_pull(env, task, drawer_name, obj_name, obj_loc)
                        result['lock_known'] = True
                        print(f"[Exploration] Pulled handle '{obj_name}' on drawer '{drawer_name}'.")
            except Exception as e:
                print(f"[Exploration][Info] Could not pull '{obj_name}': {e}")

            # (d) Try to sweep (probe for other knowledge, e.g. identified) if available 
            try:
                print(f"[Exploration] Try to sweep '{obj_name}' at '{obj_loc}'...")
                obs, reward, done = execute_sweep(env, task, obj_name, obj_loc)
                result['identified'] = True
                print(f"[Exploration] Sweep done on '{obj_name}'.")
            except Exception as e:
                print(f"[Exploration][Info] Sweep failed for '{obj_name}': {e}")

            # (e) Try to check temperature
            try:
                if 'temperature_sensor' in positions:
                    print(f"[Exploration] Checking temperature for '{obj_name}'...")
                    obs, reward, done = execute_gripper(env, task)
                    result['temperature_known'] = True
            except Exception as e:
                print(f"[Exploration][Info] Could not get temperature for '{obj_name}': {e}")

            # Store results
            check_results[obj_name] = result
            print(f"[Exploration] Finished exploration for '{obj_name}'. Results: {result}")

        # 2. Exploration with drawers (if needed for lock/durability etc.)
        for drawer_name in all_drawers:
            drawer_info = positions[drawer_name]
            drawer_loc = drawer_info.get('location', None)
            drawer_pos = drawer_info.get('position', None)
            if drawer_loc is None and drawer_pos is not None:
                drawer_loc = drawer_pos

            try:
                print(f"[Exploration] Move to drawer '{drawer_name}' at '{drawer_loc}'.")
                obs, reward, done = execute_go(env, task, robot_pose, drawer_loc)
                robot_pose = drawer_loc
            except Exception as e:
                print(f"[Exploration][Info] Could not move to '{drawer_name}': {e}")

            try:
                print(f"[Exploration] Attempting to push drawer '{drawer_name}' at '{drawer_loc}'.")
                obs, reward, done = execute_push(env, task, drawer_name, drawer_loc)
                print(f"[Exploration] Pushed '{drawer_name}'.")
            except Exception as e:
                print(f"[Exploration][Info] Could not push '{drawer_name}': {e}")

        print("[Exploration] Exploration phase complete. Check which objects reveal missing predicates via above actions and outputs.")

        # === You may now use the exploration result to decide what predicate was missing, e.g., lock-known or weight-known,
        # and continue further with your main task/plan if desired. For now, we focus only on exploring missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()