# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill/action functions!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers
        init_video_writers(obs)

        # Wrap the task steps for video recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Example: Retrieve object locations and other metadata ===
        positions = get_object_positions()  # Dictionary: {'object_1':(x,y,z), ...}

        # === EXPLORATION PHASE FOR MISSING PREDICATES ===
        print("===== Exploration Phase: Checking Predicate Availability =====")
        # We know from the domain that certain skills depend on predicates being present/absent.
        # The observation and feedback suggest that a predicate might be missing, causing planning failure.
        # We'll systematically try actions to elicit errors and discover what's missing.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep',
            'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # We will attempt to invoke available skills on some object/location.
        # This will help expose missing information (e.g. missing predicates or metadata).
        # We'll log successes or failures.

        # Select some possible objects and locations from the positions dict
        object_names = [k for k in positions.keys() if 'drawer' not in k and 'handle' not in k]
        drawer_names = [k for k in positions.keys() if 'drawer' in k]
        handle_names = [k for k in positions.keys() if 'handle' in k]
        location_names = list(set([k for k in positions.keys() if 'location' in k or 'pose' in k]))
        # If above fails (no explicit 'location' names in dict), fallback to any object 
        if not location_names:
            location_names = list(positions.keys())

        robot_location = None
        try:
            # Try to infer robot location if available in positions (common names)
            for rn in ['robot', 'robby', 'ready-pose']:
                if rn in positions:
                    robot_location = rn
                    break
        except Exception:
            robot_location = None

        # For exploration, pick a representative object and location
        test_obj = object_names[0] if object_names else None
        test_drawer = drawer_names[0] if drawer_names else None
        test_handle = handle_names[0] if handle_names else None
        test_loc = location_names[0] if location_names else None

        # Log selected test references
        print(f"[Exploration] object: {test_obj}, drawer: {test_drawer}, handle: {test_handle}, location: {test_loc}, robot_location: {robot_location}")

        # Try each skill and note if any fail due to missing predicates or other errors
        exploration_results = {}
        for skill in available_skills:
            print(f"[Exploration] Attempting skill: {skill}")
            try:
                # The signature for each skill is different, but we must use only predefined skills and known objects
                if skill == 'execute_pick':
                    # Try pick object (non-handle): needs object, location
                    if test_obj and test_loc:
                        obs, reward, done = execute_pick(env, task, obj_name=test_obj, location_name=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate object/location)'
                elif skill == 'execute_place':
                    # Try place object: needs object, drawer, location.
                    if test_obj and test_drawer and test_loc:
                        obs, reward, done = execute_place(env, task, obj_name=test_obj, drawer_name=test_drawer, location_name=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate object/drawer/location)'
                elif skill == 'execute_push':
                    # Try push drawer: needs drawer, location
                    if test_drawer and test_loc:
                        obs, reward, done = execute_push(env, task, drawer_name=test_drawer, location_name=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate drawer/location)'
                elif skill == 'execute_pull':
                    # Try pull using handle: needs drawer, handle, location
                    if test_drawer and test_handle and test_loc:
                        obs, reward, done = execute_pull(env, task, drawer_name=test_drawer, handle_name=test_handle, location_name=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate drawer/handle/location)'
                elif skill == 'execute_sweep':
                    # Try sweep object: needs object, location
                    if test_obj and test_loc:
                        obs, reward, done = execute_sweep(env, task, obj_name=test_obj, location_name=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate object/location)'
                elif skill == 'execute_rotate':
                    # Try rotate an object: needs object, location
                    if test_obj and test_loc:
                        obs, reward, done = execute_rotate(env, task, obj_name=test_obj, location_name=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate object/location)'
                elif skill == 'execute_go':
                    # Try move from one location to another
                    if robot_location and test_loc:
                        obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=test_loc)
                        exploration_results[skill] = 'Success'
                    else:
                        exploration_results[skill] = 'Skipped (no candidate from/to location)'
                elif skill == 'execute_gripper':
                    # Try gripper primitive (no arguments)
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill] = 'Success'
                else:
                    exploration_results[skill] = 'Unknown skill!'
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")
                exploration_results[skill] = f'Exception: {type(e).__name__}: {e}'
        
        print("[Exploration] Exploration results summary:")
        for k,v in exploration_results.items():
            print(f"  {k} : {v}")

        # === END OF EXPLORATION PHASE ===

        # TODO: Based on the exploration results and observation, 
        # you now know which action fails (and thus which predicate/info is missing). 
        # Next, you can choose to invoke task-specific planning or request fix to PDDL.

        print("[Exploration] Exploration phase complete. Analyze which predicate or information was missing from the failures above.")

        # (Optionally, if you want to now try a concrete plan, you may insert action sequences here
        # for the robot to solve a specific drawer-object manipulation, for example...)

        # Example placeholder for plan execution:
        # Suppose after exploration, you determined the missing predicate was 'drawer-unlocked ?d'
        # You would now request the planner or PDDL author add that fact to the initial state, then replan.

        # End of main logic

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()