# run_skeleton_task.py (Filled version for predicate exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # (primitives like execute_go, execute_pick, etc.)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, focusing on exploration to determine missing predicate per feedback.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        print("[Exploration] Starting exploration to find missing predicate 'hand-empty'.")
        # This exploration phase focuses on predicates, specifically 'hand-empty' due to the feedback.

        # Main idea:
        # - To discover if 'hand-empty' or an equivalent predicate ('handempty', 'robot-free', 'hand_empty') is needed by the environment's underlying logic or affects the skill execution.
        # - We will attempt a pick/hold action and then a subsequent pick/hold action (without placing), to see if a predicate like 'hand-empty' blocks the second action as in the PDDL definitions.
        # - This will help us determine if 'hand-empty' must be included actively in our predicates and action preconditions during planning/skill invocation.
        
        # Step 1. Get first pickable object and a location to move to, as well as their required parameters
        pickable_objects = [name for name in positions if 'handle' not in name and 'drawer' not in name]
        handles = [name for name in positions if 'handle' in name]
        drawers = [name for name in positions if 'drawer' in name]
        locations = [name for name in positions if 'pose' in name or 'location' in name or 'ready' in name]
        if len(pickable_objects) == 0:
            print("[Exploration] No pickable non-handle object found, cannot continue exploration.")
            return
        obj_to_pick = pickable_objects[0]
        location = None
        for loc_name in positions:
            if 'ready' in loc_name or 'pose' in loc_name or 'location' in loc_name:
                location = loc_name
                break
        if location is None:
            location = list(positions.keys())[0]
        # Step 2. Try to pick obj_to_pick (first time): should succeed if hand is empty
        print(f"[Exploration] Attempting to pick up {obj_to_pick} at {location}.")
        pick_success = False
        try:
            obs, reward, done = execute_pick(env, task, obj_to_pick, location)
            print(f"[Exploration] First pick success: {obs is not None}")
            pick_success = True
        except Exception as e:
            print(f"[Exploration] Exception on first pick: {str(e)}")
        # Step 3. Without placing, try to 'pick' another object to simulate a blocked precondition
        second_obj = obj_to_pick
        if len(pickable_objects) > 1:
            second_obj = pickable_objects[1]
        print(f"[Exploration] Attempting to pick up {second_obj} (again) without putting down first object.")
        try:
            obs2, reward2, done2 = execute_pick(env, task, second_obj, location)
            print(f"[Exploration] Second pick (without freeing hand) success: {obs2 is not None}")
            if obs2 is not None:
                print("[Exploration] WARNING: Second pick succeeded without explicit hand-empty check.")
            else:
                print("[Exploration] Second pick failed: possible missing 'hand-empty' precondition enforced.")
        except Exception as e2:
            print(f"[Exploration] Exception on second pick: {str(e2)}")
            print("[Exploration] This suggests that the 'hand-empty' predicate or its alias is missing or not enforced.")

        print("[Exploration] Predicate testing complete. If the second pick was blocked, a 'hand-empty' or equivalent predicate is necessary in the domain/problem.")

        # Optionally, place/back to ready pose to observe later steps (would use execute_place if available).
        # For now just end after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
