# run_skeleton_task.py (Fully Generic Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # skills: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with an exploration phase to determine the missing predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # EXPLORATION PHASE: Determine which predicate is missing using effect of "need-ready"
        # Feedback and skill names indicate that (need-ready) predicate is sometimes not managed/cleared;
        # Our exploration will involve monitoring the effect of actions that trigger or clear (need-ready).

        print("===== [Exploration Phase]: Checking for missing predicate related to (need-ready) =====")

        # We'll attempt actions that (per domain PDDL) modify "need-ready":
        # - execute_pick_object or execute_place_object set (need-ready)
        # - execute_go_ready clears (need-ready)
        #
        # Our exploration is to try a sequence that sets "need-ready" and then check if the environment
        # prevents further actions, then clears it using "execute_go_ready".
        #
        # (We assume object_positions provides at least one object and one location, along with
        #  possible drawers and handles if needed. We'll handle missing keys gracefully.)

        # Try to extract one object for picking and a location
        objects = [k for k, v in positions.items() if "object" in k or "obj" in k]
        drawers = [k for k, v in positions.items() if "drawer" in k]
        handles = [k for k, v in positions.items() if "handle" in k]
        locations = [k for k, v in positions.items() if "location" in k or "room" in k or "pose" in k]

        # Fallback: pick any keys for demonstration if typed info not present
        if not objects and positions:
            objects = [list(positions.keys())[0]]
        if not locations and positions:
            locations = [list(positions.keys())[0]]

        # Step 1: Try to execute a pick; assume generic names if extraction failed
        pick_obj = objects[0] if objects else None
        pick_loc = locations[0] if locations else None

        if pick_obj is not None and pick_loc is not None:
            try:
                print(f"[Exploration] Attempting to pick object '{pick_obj}' at location '{pick_loc}' to induce (need-ready)...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=pick_obj,
                    target_pos=positions[pick_obj],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Pick complete, (need-ready) predicate should now be set.")

                # Now try to execute another skill that is blocked by (need-ready), expect it to fail
                print("[Exploration] Trying to execute 'execute_place' while need-ready is set (should be blocked)...")
                try:
                    # Use a drawer for placing if available:
                    place_drawer = drawers[0] if drawers else pick_obj
                    obs2, reward2, done2 = execute_place(
                        env,
                        task,
                        target_obj=pick_obj,
                        target_drawer=place_drawer,
                        target_pos=positions[pick_obj],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print("[Exploration] Unexpected: Place allowed when need-ready is set (check predicates).")
                except Exception as e:
                    print(f"[Exploration] As expected, place was blocked due to (need-ready): {e}")

                # Clear (need-ready) using the ready-pose
                print("[Exploration] Executing execute_go_ready to clear (need-ready)...")
                try:
                    obs_ready, reward_ready, done_ready = execute_go(
                        env,
                        task,
                        from_pose=pick_loc,
                        to_pose='ready-pose',
                        approach_distance=0.15,
                        max_steps=50,
                        threshold=0.01,
                        timeout=10.0
                    )
                except Exception as e_go:
                    print(f"[Exploration] Unable to move to ready-pose: {e_go}")
                    obs_ready = obs

                try:
                    obs, reward, done = execute_gripper(
                        env,
                        task,
                        max_steps=20,
                        timeout=2.0
                    )
                except Exception as eg:
                    print("[Exploration] Gripper action failed (possibly due to need-ready) - this is expected during exploration.")

                print("[Exploration] Exploration run complete. (need-ready) should be reset, primitive skills unlocked.")

            except Exception as e:
                print(f"[Exploration] Error during induced need-ready exploration: {e}")
        else:
            print("[Exploration] Skipping, insufficient object and location info in positions:", positions)

        # === RESUME TASK: Continue normal execution for the oracle plan, using available skills ===
        # At this point the system has demonstrated that (need-ready) blocks skills, and is cleared by the ready-pose sequence.

        # --- Example Oracle Plan Steps ---
        # For demonstration, perform pick -> go_ready (clear need-ready) -> place, if possible

        print("\n===== [Main Plan Execution]: Oracle plan steps after resolving (need-ready) =====")
        # 1. Pick an object (if not already holding)
        if pick_obj is not None and pick_loc is not None:
            try:
                print(f"[Plan] Picking object '{pick_obj}' at location '{pick_loc}'...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=pick_obj,
                    target_pos=positions[pick_obj],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Plan] Error in pick action: {e}")

            # 2. Go to ready pose to clear (need-ready)
            try:
                print(f"[Plan] Moving to 'ready-pose' to clear (need-ready)...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_pose=pick_loc,
                    to_pose='ready-pose',
                    approach_distance=0.15,
                    max_steps=50,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Plan] Error in go to ready-pose action: {e}")

            # 3. Place the object if possible
            place_drawer = drawers[0] if drawers else None
            if place_drawer is not None:
                try:
                    print(f"[Plan] Placing object '{pick_obj}' into drawer '{place_drawer}'...")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        target_obj=pick_obj,
                        target_drawer=place_drawer,
                        target_pos=positions[pick_obj],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Plan] Error in place action: {e}")
            else:
                print("[Plan] No drawer available for place action.")
        else:
            print("[Plan] Insufficient object/location info to execute main plan.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
