# run_skeleton_task.py (Full Implementation with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine or add skill functions.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap task steps for video
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Retrieve positions (e.g., {'drawer_bottom': (x, y, z), ...})
        positions = get_object_positions()
        
        # === Setup: Extract Key Environment Entities ===
        # The names below are assumed standard; may be adjusted as required.
        # You can adjust these variables if your objects/locations are named differently in your RLBench scenario.
        robot_current_location = None
        drawer_name = None
        drawer_loc = None
        drawer_handle = None

        # Attempt to identify the main drawer and handle in the scene
        # This block tries to find a likely candidate for the drawer and its handle
        for obj_name in positions:
            if 'drawer' in obj_name:
                if 'handle' in obj_name:
                    drawer_handle = obj_name
                else:
                    drawer_name = obj_name
                    drawer_loc = obj_name  # Assuming they share the name/location

        if drawer_name is None:
            # Fallback if the above generic search fails
            for obj_name in positions:
                if obj_name.startswith('drawer'):
                    drawer_name = obj_name
                    drawer_loc = obj_name

        if drawer_handle is None:
            # Try to find a handle for the drawer
            for obj_name in positions:
                if 'handle' in obj_name:
                    drawer_handle = obj_name

        if drawer_name is None or drawer_loc is None:
            print("[Error] Could not identify drawer in the environment.")
            return

        # Attempt to determine starting robot position
        robot_possible_keys = ['robot', 'agent', 'base', 'ready-pose', 'ready_pose', 'ready']
        for name in positions:
            for key in robot_possible_keys:
                if key in name:
                    robot_current_location = name
                    break
            if robot_current_location is not None:
                break
        
        if robot_current_location is None:
            # Fallback: just pick any key if not found
            robot_current_location = list(positions.keys())[0]

        print(f"[Env] Detected robot position: {robot_current_location}")
        print(f"[Env] Detected drawer: {drawer_name} at {drawer_loc}")
        print(f"[Env] Detected drawer handle: {drawer_handle}")

        # === Exploration Phase ===
        # (Goal: Discover the "drawer-open drawer_bottom" state by trying to open the drawer.)

        # Context: From feedback, we know opening the drawer should result in the (drawer-open drawer_bottom) predicate.
        # We perform the high-level sequence of skills using only available skill names.

        # Skills available: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Skill: Move to drawer location
        try:
            print("[Exploration] Moving to drawer location...")
            # Assume robot_current_location is the current, drawer_loc is desired
            obs, reward, done, info = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location=drawer_loc
            )
            robot_current_location = drawer_loc  # Update after move
        except Exception as e:
            print(f"[Warning] Problem during execute_go: {e}")

        # Skill: Pick up the handle (if handle is defined)
        picked_handle = False
        if drawer_handle is not None:
            try:
                print(f"[Exploration] Attempting to pick up handle: {drawer_handle}")
                obs, reward, done, info = execute_pick(
                    env,
                    task,
                    obj_name=drawer_handle,
                    location=drawer_loc
                )
                picked_handle = True
            except Exception as e:
                print(f"[Warning] Could not pick handle '{drawer_handle}'. Error: {e}")

        # Skill: Pull the drawer (assuming pick succeeded)
        if picked_handle:
            try:
                print(f"[Exploration] Pulling the drawer: {drawer_name} using handle {drawer_handle}")
                obs, reward, done, info = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=drawer_handle,
                    location=drawer_loc
                )
                
                # If successful, drawer should now be open.
                print(f"[Exploration] Drawer '{drawer_name}' should now be open! (Expect predicate: (drawer-open {drawer_name}))")

            except Exception as e:
                print(f"[Warning] Problem during execute_pull: {e}")
        else:
            print(f"[Exploration] Did not succeed in picking up handle '{drawer_handle}'; cannot pull.")

        # Optionally: You could explore further skills here to discover more predicates/state, but our main focus is to reveal the (drawer-open drawer_bottom) state.

        # === REPORT: Explored Predicate ===
        # Since we've performed the pull, we expect "(drawer-open drawer_bottom)" should now be true.
        print("[Result] Exploration attempted: the missing predicate should be (drawer-open drawer_bottom).")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
