# run_skeleton_task.py (Filled to perform exploration for missing predicate and use only provided skills)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-defined skills, do not define new ones

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --------------------------------------------------------------------------
        # === Exploration Phase to Find Missing Predicate (drawer-open drawer_bottom)
        # --------------------------------------------------------------------------

        # Step 1: Identify available drawers and their states (in positions dict)
        # Let's try to find 'drawer_bottom', or fallback to any drawer.
        drawer_candidates = [name for name in positions if 'drawer' in name]
        if len(drawer_candidates) == 0:
            print("[Error] No drawers found in object positions!")
            return

        # Try to use 'drawer_bottom' if available
        if 'drawer_bottom' in positions:
            drawer_name = 'drawer_bottom'
        else:
            drawer_name = drawer_candidates[0]
            print(f"[Warning] 'drawer_bottom' not in positions, using '{drawer_name}'.")

        drawer_pos = positions[drawer_name]
        robot_pos_keys = [k for k in positions if 'robot' in k or 'base' in k or 'ready' in k]
        if 'ready-pose' in positions:
            robot_init_pos_key = 'ready-pose'
        elif len(robot_pos_keys) > 0:
            robot_init_pos_key = robot_pos_keys[0]
        else:
            # Use any position as fallback
            robot_init_pos_key = list(positions.keys())[0]

        robot_init_pos = positions[robot_init_pos_key]

        # Assume these skills are present per provided list
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Step 2: Attempt to open the drawer using provided skills.
        # But if (drawer-open drawer_bottom) predicate is missing, the action may fail.
        #
        # Exploration: try to execute the standard skills and see what fails, to learn what predicate is missing.

        print(f"[Exploration] Attempting to open '{drawer_name}' (at position {drawer_pos})...")

        # We assume handles exist; try to find the handle associated if present. For robustness, select any handle object.
        handle_name = None
        for obj in positions:
            if 'handle' in obj:
                handle_name = obj
                break
        # If can't find, just use any 'object' that is not the drawer
        if handle_name is None:
            handle_candidates = [o for o in positions if o != drawer_name]
            if len(handle_candidates) > 0:
                handle_name = handle_candidates[0]
            else:
                print('[Error] No handle/object found for pulling drawer!')
                return
        handle_pos = positions[handle_name]

        # Move robot to drawer position if not already there
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=robot_init_pos,
                to_pos=drawer_pos
            )
        except Exception as e:
            print(f"[Exception] Failed to move to drawer position: {e}")

        # Pick up the handle (simulate 'execute_pick')
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=handle_name,
                location=drawer_pos
            )
            print(f"[Exploration] Picked handle '{handle_name}' for drawer '{drawer_name}'.")
        except Exception as e:
            print(f"[Exception] execute_pick failed: {e}")

        # Try to pull the drawer (simulate 'execute_pull')
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_pos
            )
            print(f"[Exploration] Pulled to open '{drawer_name}'. If this failed, likely the drawer is locked or already open.")
        except Exception as e:
            print(f"[Exception] execute_pull failed: {e}")

        # Check for 'drawer-open' predicate
        # Since we don't have direct PDDL predicate checking here, you would normally
        # check if the visual state or object pose indicates an open drawer.
        # For this skeleton, report that we attempted to pull.

        print(f"[Exploration] If execute_pull fails or drawer state does not change to 'open', suspect missing predicate: (drawer-open {drawer_name})")

        # --------------------------------------------------------------------------
        # === Main Oracle Plan Execution Phase (after exploration)
        # --------------------------------------------------------------------------
        #
        # You would continue with the oracle plan here, using the provided skills
        # such as execute_place, execute_push, execute_pick, etc., following the
        # planning logic using the environment object IDs and positions.
        #
        # For this template, we demonstrate up to the exploration and predicate
        # identification as required by feedback.
        #

        # TODO: Insert plan execution logic here as per your oracle plan,
        # using only predefined skills from skill_code.
        #

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
