# run_skeleton_task.py (Completed with Exploration for need-ready predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Positions is assumed to be a dict mapping object names to their (x,y,z) positions.
        positions = get_object_positions()

        # === Exploration Phase for need-ready ===

        print("[Exploration] Starting predicate exploration to locate missing predicate (need-ready) in plan execution.")
        # Assumption: We want to trigger an action that sets the need-ready predicate in the world state and observe the effect on available actions.

        # Let's attempt to use 'execute_pick' and then try to immediately use another skill to see if need-ready blocks it (simulate missing predicate issue).

        # Here, we simulate a basic "explore" by attempting to pick an object and then immediately do a place, push, or move,
        # watching for failure or blocked action due to need-ready.

        # 1. Get a candidate object on the floor (non-handle) and its position
        picked = False
        placed = False
        pushed = False
        ready_pos_done = False
        
        object_on_floor = None
        object_pos = None
        for obj_name, pos in positions.items():
            # This is a heuristic: typical handles are named 'handle' or similar. We avoid them.
            # (More robust: if get_object_positions() or observation can tell us which is handle.)
            if 'handle' not in obj_name and 'drawer' not in obj_name:
                object_on_floor = obj_name
                object_pos = pos
                break
        
        if object_on_floor is None:
            print("[Exploration] No appropriate object found on floor for pick. Please check the environment setup.")
        else:
            # Find a location for the robot (guess nearest or use known 'ready-pose' or whatever's in obs)
            try:
                # If 'robot' or 'agent' is in the positions, pick its current location
                robot_pos = None
                for k in positions:
                    if 'robot' in k or 'agent' in k:
                        robot_pos = positions[k]
                        break
                if robot_pos is None:
                    # fallback default pose (could be improved)
                    robot_pos = [0., 0., 0.]

                # === Execute: Pick the object (should set need-ready) ===
                # Use execute_pick for object
                print(f"[Exploration] Attempting to pick object '{object_on_floor}' at position {object_pos}")
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        target=object_on_floor,
                        location=object_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    picked = True
                    print("[Exploration] Successfully executed pick. Testing predicate effect of need-ready...")

                except Exception as e:
                    print(f"[Exploration] Exception during pick: {e}")
                
                # 2. Try to immediately call another skill: try execute_place, execute_push, or execute_go
                # With need-ready set, further actions (other than go_ready) should be blocked.
                # We'll check for error or blocked action.

                if picked:
                    # Try execute_place (will likely fail if need-ready is present)
                    try:
                        # Find a drawer to place into
                        candidate_drawer = None
                        drawer_pos = None
                        for name in positions:
                            if 'drawer' in name and 'handle' not in name:
                                candidate_drawer = name
                                drawer_pos = positions[name]
                                break
                        if candidate_drawer is not None:
                            print(f"[Exploration] Attempting to place object into '{candidate_drawer}'")
                            obs2, reward2, done2 = execute_place(
                                env,
                                task,
                                object_name=object_on_floor,
                                drawer_name=candidate_drawer,
                                location=drawer_pos,
                                approach_distance=0.10,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            placed = True
                            print("[Exploration] Place unexpectedly succeeded. 'need-ready' may not block place as expected.")
                        else:
                            print("[Exploration] No candidate drawer found to test place.")

                    except Exception as e:
                        print("[Exploration] Exception during place after pick (likely predicate block due to need-ready):", str(e))
                        print("[Exploration Result] Blocked after pick: likely missing predicate 'need-ready' causes this.")
                    
                    # Try execute_push
                    if not placed:
                        try:
                            candidate_drawer_push = candidate_drawer
                            if candidate_drawer_push is not None:
                                print(f"[Exploration] Attempting to push '{candidate_drawer_push}'")
                                obs3, reward3, done3 = execute_push(
                                    env,
                                    task,
                                    drawer_name=candidate_drawer_push,
                                    location=drawer_pos,
                                    max_steps=30,
                                    approach_distance=0.07,
                                    timeout=7.0
                                )
                                pushed = True
                                print("[Exploration] Push unexpectedly succeeded. 'need-ready' may not block push as expected.")
                        except Exception as e:
                            print("[Exploration] Exception during push after pick:", str(e))
                    
                    # Try execute_go (move somewhere else)
                    if not placed and not pushed:
                        try:
                            some_location = 'ready-pose'
                            print(f"[Exploration] Attempting to move to '{some_location}'")
                            obs4, reward4, done4 = execute_go(
                                env,
                                task,
                                from_location=robot_pos,
                                to_location=some_location,
                                max_steps=30,
                                timeout=10.0
                            )
                            print("[Exploration] Move to ready-pose attempted.")
                        except Exception as e:
                            print("[Exploration] Exception during execute_go after pick:", str(e))
                
                # 3. Call execute_go_ready to clear need-ready
                try:
                    print("[Exploration] Attempting to call execute_go_ready to clear 'need-ready' flag")
                    obs5, reward5, done5 = execute_go_ready(
                        env,
                        task,
                        from_location=robot_pos,
                        max_steps=30,
                        timeout=10.0
                    )
                    ready_pos_done = True
                    print("[Exploration] Successfully executed execute_go_ready. 'need-ready' predicate should be cleared.")

                except Exception as e:
                    print("[Exploration] Exception during execute_go_ready:", str(e))
                
                # 4. Try picking or placing again to confirm need-ready is cleared
                if ready_pos_done:
                    try:
                        print(f"[Exploration] Attempting to pick '{object_on_floor}' again after go_ready.")
                        obs6, reward6, done6 = execute_pick(
                            env,
                            task,
                            target=object_on_floor,
                            location=object_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print("[Exploration] Pick after go_ready succeeded. Confirmed: 'need-ready' predicate blocks actions until cleared by go_ready.")
                    except Exception as e:
                        print("[Exploration] Unexpected issue: Could not pick after execute_go_ready.", str(e))

            except Exception as e:
                print(f"[Exploration] Error in exploration logic: {e}")

        print("===== End of Exploration Phase. If predicate was missing, above should have revealed necessary info. =====")
        
        # === Normal Plan Execution would follow here (not implemented for exploration-only run) ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()