# run_skeleton_task.py (Completed for Exploration - need-ready)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all provided skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Now positions contains mapping like: {'object1': pos1, 'drawer1': pos2, ...}

        # == EXPLORATION PHASE - Identify predicates that might be missing ==
        print("[Exploration] Starting exploration to reveal hidden/missing predicates (e.g., 'need-ready')")
        # We'll systematically call available skills and watch for failures when 'need-ready' is set

        skill_names = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        exploration_results = {}
        step_result_map = {}

        # Try all skills one by one and record if hitting a "blocked" state due to (need-ready)
        # Here we use dummy objects/positions as skill interface - adapt keys from positions dict
        # For minimal demonstration, use only keys existing within 'positions' dict and catch failures
        
        for skill in skill_names:
            func = globals().get(skill, None)
            if func is None:
                exploration_results[skill] = 'NOT_FOUND'
                continue
            # Prepare arguments according to canonical names (best effort)
            try:
                # Heuristic for arguments based on skill name
                if skill == 'execute_pick':
                    # Try to pick up the first object found in positions (not a drawer or handle)
                    test_obj = None
                    for k in positions:
                        if 'drawer' not in k and 'handle' not in k:
                            test_obj = k
                            break
                    # Assume pick-location is where the object is
                    if test_obj:
                        loc = positions[test_obj]
                        res = func(env, task, obj_name=test_obj, pos=loc)
                        step_result_map[skill] = res
                        exploration_results[skill] = 'SUCCESS'
                    else:
                        exploration_results[skill] = 'NO_VALID_OBJECT'
                elif skill == 'execute_place':
                    # Try to place first held object into first drawer at its position
                    # Simulate: find object, drawer, and location
                    obj_name = None
                    drawer_name = None
                    for k in positions:
                        if 'drawer' in k and drawer_name is None:
                            drawer_name = k
                        if 'drawer' not in k and obj_name is None:
                            obj_name = k
                    if obj_name and drawer_name:
                        loc = positions[drawer_name]
                        res = func(env, task, obj_name=obj_name, drawer_name=drawer_name, pos=loc)
                        step_result_map[skill] = res
                        exploration_results[skill] = 'SUCCESS'
                    else:
                        exploration_results[skill] = 'INSUFFICIENT_OBJECTS'
                elif skill == 'execute_push':
                    # Try to push first drawer
                    drawer_name = None
                    for k in positions:
                        if 'drawer' in k and drawer_name is None:
                            drawer_name = k
                    if drawer_name:
                        loc = positions[drawer_name]
                        res = func(env, task, drawer_name=drawer_name, pos=loc)
                        step_result_map[skill] = res
                        exploration_results[skill] = 'SUCCESS'
                    else:
                        exploration_results[skill] = 'NO_DRAWER'
                elif skill == 'execute_pull':
                    # Try to execute pull using handle if available
                    drawer_name = None
                    handle_name = None
                    for k in positions:
                        if 'handle' in k and handle_name is None:
                            handle_name = k
                        if 'drawer' in k and drawer_name is None:
                            drawer_name = k
                    if drawer_name and handle_name:
                        loc = positions[drawer_name]
                        res = func(env, task, drawer_name=drawer_name, handle_name=handle_name, pos=loc)
                        step_result_map[skill] = res
                        exploration_results[skill] = 'SUCCESS'
                    else:
                        exploration_results[skill] = 'NO_HANDLE_OR_DRAWER'
                elif skill == 'execute_sweep' or skill == 'execute_rotate':
                    # Arbitrary object and pos
                    test_obj = next(iter(positions), None)
                    if test_obj:
                        loc = positions[test_obj]
                        res = func(env, task, obj_name=test_obj, pos=loc)
                        step_result_map[skill] = res
                        exploration_results[skill] = 'SUCCESS'
                    else:
                        exploration_results[skill] = 'NO_OBJECT'
                elif skill == 'execute_go':
                    # Move from one location to another (if possible)
                    # Get two different objects as proxies for locations
                    loc_keys = list(positions.keys())
                    if len(loc_keys) >= 2:
                        from_key = loc_keys[0]
                        to_key = loc_keys[1]
                        from_pos = positions[from_key]
                        to_pos = positions[to_key]
                        res = func(env, task, from_pos=from_pos, to_pos=to_pos)
                        step_result_map[skill] = res
                        exploration_results[skill] = 'SUCCESS'
                    else:
                        exploration_results[skill] = 'NO_TWO_LOCATIONS'
                elif skill == 'execute_gripper':
                    res = func(env, task)
                    step_result_map[skill] = res
                    exploration_results[skill] = 'SUCCESS'
                else:
                    exploration_results[skill] = 'UNKNOWN_ARGS'
            except Exception as ex:
                # If the exception or result signals "blocked by need-ready", note it
                msg = str(ex)
                if "need-ready" in msg or "'need-ready'" in msg or "blocked" in msg:
                    exploration_results[skill] = 'BLOCKED_BY_NEED_READY'
                else:
                    exploration_results[skill] = f'EXCEPTION: {msg}'

        print("[Exploration] Exploration Results by Skill:")
        for skill, result in exploration_results.items():
            print(f"   - {skill}: {result}")

        # Based on the feedback, if we observe that many/all skills are blocked by need-ready except execute_go_ready,
        # We deduce that (need-ready) is the hidden predicate gating skill availability after pick/place

        # === SUMMARY OUTPUT ===
        print("\n[Summary] Exploration complete.")
        print("[Summary] Missing Predicate likely responsible for blocking is: (need-ready)")

        # The user can now proceed to either reset the environment or insert explicit calls to
        # 'execute_go_ready' at appropriate points in the main solution plan after pick/place actions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
