# run_skeleton_task.py (Completed Task Execution including Exploration for Missing Predicate 'need-ready')

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to locations/tuples
        positions = get_object_positions()

        # === Exploration Phase ===
        # Based on feedback, explore missing predicate (need-ready) via available skills and domain
        # The missing predicate 'need-ready' is triggered/set by certain actions (e.g., execute_pick_object, execute_place_object)
        # If the plan fails at actions blocked by (need-ready), we must clear it using execute_go_ready

        # For demonstration, let's assume we have a TEST OBJECT (like 'obj1'), a DRAWER (like 'drawer1'), and POSITIONS inferred
        # You may modify object/drawer/handle names and positions for your real scenario.

        # Specify your object, drawer, handle names and their positions by examining positions or your setup
        # Example key names (replace per your env): 'obj1', 'drawer1', 'handle1', 'ready-pose', etc.
        try:
            # Infer object, drawer, handle, location names from positions
            obj_name = None
            drawer_name = None
            handle_name = None
            location_names = set()
            for k, v in positions.items():
                if 'drawer' in k:
                    drawer_name = k
                elif 'handle' in k:
                    handle_name = k
                elif 'obj' in k or 'ball' in k or 'cube' in k:
                    obj_name = k
                else:
                    # Possibly a location name
                    location_names.add(k)
            # Fallback dummy names if not inferred
            if not obj_name:
                obj_name = 'obj1'
            if not drawer_name:
                drawer_name = 'drawer1'
            if not handle_name:
                handle_name = 'handle1'
            # Locations for robot to move to; pick first two locations in dictionary
            location_list = [k for k in positions.keys() if 'pose' in k or 'loc' in k or 'position' in k or 'ready' in k]
            if len(location_list) < 2:
                # fallback
                location_list = ['A', 'B']
            loc1 = location_list[0]
            loc2 = location_list[1] if len(location_list) > 1 else location_list[0]

        except Exception as e:
            print("[Exploration] Failed to infer names from positions:", e)
            obj_name = 'obj1'
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            loc1 = 'A'
            loc2 = 'B'

        print(f"[Exploration] Using object: {obj_name}, drawer: {drawer_name}, handle: {handle_name}, locations: {loc1}, {loc2}")

        # Step 1: Try 'execute_pick' or 'execute_place', expecting 'need-ready' becomes True and blocks further actions
        # Try picking an object; if afterward 'need-ready' is set, must call 'execute_go_ready' before continuing

        try:
            print("[Exploration] Attempting execute_pick on object:", obj_name, "at", loc1)
            obs, reward, done = execute_pick(env, task, obj_name, loc1)
            print("[Exploration] After execute_pick. (If blocked in next step, 'need-ready' is required.)")
        except Exception as e:
            print("[Exploration][Error] execute_pick failed:", e)

        # Attempt next action that would be blocked by (need-ready)
        try:
            print("[Exploration] Attempting execute_place (may be blocked by 'need-ready') on object", obj_name, "into", drawer_name, "at", loc2)
            obs, reward, done = execute_place(env, task, obj_name, drawer_name, loc2)
            print("[Exploration] execute_place completed. If blocked, 'need-ready' must be cleared.")
        except Exception as e:
            print("[Exploration][Blocked] execute_place failed:", e)
            print("[Exploration] Will attempt to clear 'need-ready' using execute_go_ready.")

            # Use execute_go_ready to clear (need-ready)
            try:
                print("[Exploration] Attempting execute_go_ready at", loc2)
                obs, reward, done = execute_go_ready(env, task, loc2)
                print("[Exploration] execute_go_ready successful. Now retrying execute_place.")
                
                # Now retry execute_place
                try:
                    obs, reward, done = execute_place(env, task, obj_name, drawer_name, loc2)
                    print("[Exploration] execute_place succeeded after clearing need-ready.")
                except Exception as e2:
                    print("[Exploration][Error-after-go_ready] execute_place still failed:", e2)

            except Exception as e_ready:
                print("[Exploration][Error] execute_go_ready failed:", e_ready)

        # After confirming that 'need-ready' must be cleared, further plan steps would always:
        # After any action that sets 'need-ready' --> must call execute_go_ready before actions needing robot-free/not (need-ready)

        # === PLAN EXECUTION PHASE ===
        # Here one would now construct a sequence:
        #   1) execute_pick (sets need-ready)
        #   2) execute_go_ready (clears need-ready)
        #   3) execute_place (sets need-ready)
        #   4) execute_go_ready (clears need-ready)
        #   ... etc. for other actions

        print("[Plan] Starting sample oracle plan execution to demonstrate need-ready handling.")

        try:
            # 1. Pick up the object
            obs, reward, done = execute_pick(env, task, obj_name, loc1)
            print("[Plan] execute_pick completed.")

            # Must clear need-ready before next constrained action
            obs, reward, done = execute_go_ready(env, task, loc1)
            print("[Plan] execute_go_ready completed.")

            # 2. Place object into drawer
            obs, reward, done = execute_place(env, task, obj_name, drawer_name, loc2)
            print("[Plan] execute_place completed.")

            # Must clear need-ready after place before further actions
            obs, reward, done = execute_go_ready(env, task, loc2)
            print("[Plan] execute_go_ready completed. Plan can continue.")

        except Exception as e:
            print("[Plan][Error] Plan execution failed:", e)

        print("===== Exploration and need-ready handling demonstration complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
