# run_skeleton_task.py (Completed with Exploration for Missing Predicate: (robot-free))

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Assumed to import all available predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We do not know exact object/position names; we just retrieve what we can

        # === EXPLORATION PHASE: Identify missing predicate/problem via exploration skills ===

        print("[Exploration] Starting exploration to find missing predicate related to 'robot-free'.")

        # In exploration, we try to perform skills and observe what fails or is blocked due to an unknown predicate.
        # Here, the feedback indicates that (robot-free) is the missing predicate.
        # We'll attempt to run all possible skills and log the outcome, looking for when a 'robot-free' precondition is required.

        exploration_skills = ['execute_go', 'execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_gripper']
        
        # For demonstration/robustness, iterate over all available skill names and try to execute each one with potentially valid arguments
        # We'll catch all exceptions and print which actions fail

        # Mapping of skill argument patterns based on skill name as in the PDDL domain
        skill_arg_patterns = {
            'execute_go':    ['from_loc', 'to_loc'],
            'execute_pick':  ['obj', 'loc'],
            'execute_place': ['obj', 'drawer', 'loc'],
            'execute_push':  ['drawer', 'loc'],
            'execute_pull':  ['drawer', 'handle', 'loc'],
            'execute_sweep': ['obj', 'loc'],
            'execute_gripper': []
        }

        # Try to find at least one location, object, drawer, and handle from available positions
        location_candidates = []
        object_candidates = []
        drawer_candidates = []
        handle_candidates = []
        for name in positions.keys():
            if 'loc' in name or 'pose' in name or 'location' in name or 'table' in name:  # heuristics
                location_candidates.append(name)
            elif 'drawer' in name:
                drawer_candidates.append(name)
            elif 'handle' in name:
                handle_candidates.append(name)
            else:
                object_candidates.append(name)

        if not location_candidates and positions:
            # Fallback: pick any
            location_candidates = list(positions.keys())
        
        # Ensure there are at least two locations for 'execute_go'
        from_loc = location_candidates[0] if len(location_candidates) > 0 else None
        to_loc = location_candidates[1] if len(location_candidates) > 1 else from_loc

        obj = object_candidates[0] if len(object_candidates) > 0 else None
        drawer = drawer_candidates[0] if len(drawer_candidates) > 0 else None
        handle = handle_candidates[0] if len(handle_candidates) > 0 else None
        loc = from_loc

        # Exploration execution
        for skill in exploration_skills:
            fn = globals().get(skill, None)
            if not fn:
                print(f"[Exploration] Skill function {skill} not found in skill_code.")
                continue
            try:
                print(f"[Exploration] Attempting skill '{skill}'...")
                if skill == 'execute_go' and from_loc and to_loc:
                    obs, reward, done = fn(env, task, from_location=from_loc, to_location=to_loc)
                elif skill == 'execute_pick' and obj and loc:
                    obs, reward, done = fn(env, task, obj=obj, location=loc)
                elif skill == 'execute_place' and obj and drawer and loc:
                    obs, reward, done = fn(env, task, obj=obj, drawer=drawer, location=loc)
                elif skill == 'execute_push' and drawer and loc:
                    obs, reward, done = fn(env, task, drawer=drawer, location=loc)
                elif skill == 'execute_pull' and drawer and handle and loc:
                    obs, reward, done = fn(env, task, drawer=drawer, handle=handle, location=loc)
                elif skill == 'execute_sweep' and obj and loc:
                    obs, reward, done = fn(env, task, obj=obj, location=loc)
                elif skill == 'execute_gripper':
                    obs, reward, done = fn(env, task)
                else:
                    print(f"[Exploration] Insufficient arguments for {skill}, skipping.")
                    continue
                print(f"[Exploration] Skill '{skill}' executed successfully or at least called.")
            except Exception as ex:
                print(f"[Exploration] Skill '{skill}' failed with exception: {repr(ex)}")
                if 'robot-free' in str(ex):
                    print("[Exploration] Detected missing predicate: (robot-free). Problem requires 'robot-free' in the preconditions.")

        # Summary output of exploration
        print("[Exploration] Exploration complete. Identified missing predicate is likely (robot-free) as required by action preconditions.")
        
        # === (Optional) Move to a ready-pose or demonstrate additional skill execution (not required here) ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()