# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills - do not reimplement primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Runs the robot task, including an exploration phase to identify the missing predicate regarding the drawer lock state.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Assumes get_object_positions returns a dict with names mapping to positions/locations.
        positions = get_object_positions()

        # You may change these keys depending on your environment setup
        # Identify relevant objects for drawer lock exploration ("drawer_bottom" as per feedback)
        drawer_name = 'drawer_bottom'
        handle_name = None
        # Search for a handle attached to this drawer
        for obj in positions.keys():
            if 'handle' in obj and drawer_name in obj:
                handle_name = obj
                break

        if handle_name is None:
            # Fallback: pick any available handle if not found
            for obj in positions.keys():
                if 'handle' in obj:
                    handle_name = obj
                    break

        if drawer_name not in positions:
            print(f"[Exploration] ERROR: '{drawer_name}' not found in object_positions.")
            return
        if handle_name is None:
            print("[Exploration] ERROR: No handle found in object_positions.")
            return

        # Guess at robot starting location; use available keys
        robot_location = None
        for k in positions.keys():
            if 'ready' in k or 'start' in k or 'pose' in k:
                robot_location = k
                break
        if robot_location is None:
            # Fallback to first location
            for k in positions.keys():
                if 'location' in k or 'room' in k or 'pose' in k:
                    robot_location = k
                    break

        # Destination location for the drawer (guess: drawer's location value)
        drawer_location = positions.get(drawer_name, None)
        if drawer_location is None:
            print(f"[Exploration] ERROR: Could not determine the drawer's location.")
            return

        # Attempt to "move" the robot to the drawer's vicinity
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_name,  # if required to use names, else provide positions/dict
                max_steps=100,
                threshold=0.02,
                timeout=10.0
            )
            if done:
                print("[Exploration] Moved to drawer location.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Attempt to pick up the handle to try and manipulate the drawer
        try:
            # Pick up handle (assume handle is on-floor and robot at drawer location)
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location_name=drawer_name,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Picked up handle:", handle_name)
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # Try to pull the drawer - this should detect lock state, updating knowledge about drawer lock
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location_name=drawer_name,
                max_steps=100,
                threshold=0.02,
                timeout=10.0
            )
            if done:
                print("[Exploration] Attempted to pull drawer to discover lock status.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")

        # Based on feedback, the exploration run will confirm the predicate
        print(f"[Exploration] Based on feedback, discovered predicate: (drawer-unlocked {drawer_name})")
        print("[Exploration] Exploration phase complete. Code ready for main oracle plan execution.")

        # --- Place your main oracle plan using available skills after exploration as needed ---

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
