# run_skeleton_task.py (Executable version with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset task to its initial state and get initial observation
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # Get initial positions and object/drawer info
        positions = get_object_positions()
        # (You may inspect and print `positions` for development/debug)
        
        # == EXPLORATION PHASE: Try available skills to elicit missing predicate feedback ==
        # In this phase, we attempt skills on available objects/locations to receive planner/environment feedback.
        # The aim is to identify missing predicates or preconditions in our PDDL domain.
        # (You may adjust object/handle/drawer names to match your env)
        object_keys = [k for k in positions.keys() if "object" in k]
        drawer_keys = [k for k in positions.keys() if "drawer" in k]
        handle_keys = [k for k in positions.keys() if "handle" in k]
        location_keys = [k for k in positions.keys() if "location" in k or k.startswith("loc")]

        # Use all available skill names
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        
        print("[Exploration] Attempting available skills to probe for missing predicates/preconditions...")
        
        exploration_attempted = False
        # Try moving between available locations
        if "execute_go" in available_skills and len(location_keys) >= 2:
            from_loc = location_keys[0]
            to_loc = location_keys[1]
            print(f"[Exploration] execute_go from {from_loc} to {to_loc}")
            try:
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] execute_go succeeded between two locations.")
                exploration_attempted = True
            except Exception as e:
                print("[Exploration] execute_go failed with exception:", str(e))
        
        # Try pick skill on an object on the floor (not a handle)
        if "execute_pick" in available_skills and object_keys:
            target_obj = object_keys[0]
            obj_pos = positions[target_obj]
            # Assume first available location is valid
            if location_keys:
                robot_loc = location_keys[0]
                print(f"[Exploration] execute_pick {target_obj} at {robot_loc}")
                try:
                    obs, reward, done = execute_pick(env, task, target_obj, robot_loc)
                    print("[Exploration] execute_pick succeeded.")
                    exploration_attempted = True
                except Exception as e:
                    print("[Exploration] execute_pick failed with exception:", str(e))

        # Try pull on any handle if available
        if "execute_pull" in available_skills and handle_keys and drawer_keys:
            target_handle = handle_keys[0]
            target_drawer = drawer_keys[0]
            if location_keys:
                current_loc = location_keys[0]
                print(f"[Exploration] execute_pull on {target_drawer} with handle {target_handle} at {current_loc}")
                try:
                    obs, reward, done = execute_pull(env, task, target_drawer, target_handle, current_loc)
                    print("[Exploration] execute_pull succeeded.")
                    exploration_attempted = True
                except Exception as e:
                    print("[Exploration] execute_pull failed with exception:", str(e))
        
        # Try sweep as a fallback exploration
        if not exploration_attempted and "execute_sweep" in available_skills and object_keys:
            if location_keys:
                robot_loc = location_keys[0]
                print(f"[Exploration] execute_sweep on {object_keys[0]} at {robot_loc}")
                try:
                    obs, reward, done = execute_sweep(env, task, object_keys[0], robot_loc)
                    print("[Exploration] execute_sweep succeeded.")
                except Exception as e:
                    print("[Exploration] execute_sweep failed with exception:", str(e))

        # Exploration done - TODO: Analyze feedback/logs to determine missing predicate(s)

        # == MAIN PLAN PHASE ==
        # Here, after collecting feedback or resolving missing predicate(s), execute the plan as per specification.
        # You must adjust the actual plan steps below per your PDDL plan and objects -- this is a generic template.
        # For demonstration, an example sequence is illustrated:

        print("[Plan] Executing oracle plan (example sequence follows; adapt per your plan).")

        try:
            # Example step 1: Move to initial object location
            if location_keys and len(location_keys) >= 2:
                from_loc, to_loc = location_keys[0], location_keys[1]
                print(f"[Plan] Robot moving from {from_loc} to {to_loc}.")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
            # Example step 2: Pick up first object
            if object_keys:
                print(f"[Plan] Picking up {object_keys[0]} at {to_loc}.")
                obs, reward, done = execute_pick(env, task, object_keys[0], to_loc)
            # Example step 3: Place the object into first available drawer, if exists
            if drawer_keys and "execute_place" in available_skills:
                print(f"[Plan] Placing {object_keys[0]} into {drawer_keys[0]} at {to_loc}.")
                obs, reward, done = execute_place(env, task, object_keys[0], drawer_keys[0], to_loc)
            # Example step 4: Return to ready position if such exists (in your env)
            if "ready-pose" in location_keys:
                print(f"[Plan] Moving to ready-pose.")
                obs, reward, done = execute_go(env, task, to_loc, "ready-pose")
        except Exception as e:
            print("[Plan] Exception during main plan execution:", str(e))
        
        # --- Add more plan steps here as needed per your plan semantics and domain objects ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
