# run_skeleton_task.py (Filled-in Executable Plan with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
# (You don't need to redefine primitives like move, pick, place. Use those from skill_code.)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Let's assume that positions keys include all relevant objects and locations:
        # e.g., positions = {'robot':..., 'drawer_1':..., 'handle_1':..., 'obj':..., 'ready-pose':..., ...}
        # For genericity, we will attempt to get handles to likely names.
        
        # Try to fetch relevant named entities safely:
        robot_pos = positions.get('robot', None)
        drawer_pos = positions.get('drawer_1', None)
        handle_pos = positions.get('handle_1', None)
        ready_pose = positions.get('ready-pose', None)
        workspace_pos = positions.get('workspace', None)

        # The PDDL suggests the predicate we need to explore is (drawer-unlocked drawer_1).
        # Before we can execute e.g. any drawer manipulation, we need to positively confirm whether the drawer is unlocked.
        # This will be the 'exploration phase'.

        print("[Task] === BEGIN EXPLORATION PHASE ===")
        drawer_is_unlocked = False
        # We'll try to use the available skills and state feedback to explore the missing predicate.
        # We attempt a skill that would fail if the drawer is locked and succeed if it is unlocked.
        #
        # Based on the domain, to check if a drawer is unlocked, we may try to pull the drawer using 'execute_pull'.
        # The 'execute_pull' precondition is (drawer-unlocked), so if it succeeds, the drawer was unlocked.

        # Prerequisite: We need to be at the relevant location and holding the handle.

        try:
            # If not at the drawer, move there first (use 'execute_go').
            # Check where the robot currently is
            robot_current_loc = positions.get('robot_location', None)
            drawer_location = positions.get('drawer_1_location', None)
            if robot_current_loc is not None and drawer_location is not None and robot_current_loc != drawer_location:
                print("[Task][Exploration] Moving robot to drawer location ...")
                obs, reward, done, info = execute_go(env, task, from_loc=robot_current_loc, to_loc=drawer_location)
                positions = get_object_positions()  # update after move
            
            # Pick handle if not already holding
            # Assume handle name is 'handle_1', and it is at the drawer location
            print("[Task][Exploration] Attempting to pick handle ...")
            obs, reward, done, info = execute_pick(env, task, obj='handle_1', loc=drawer_location)
            # Note: If already holding, this should be handled by the skill's own precondition

            # Try pulling the drawer; if the drawer is locked, this may fail gracefully or be observable.
            print("[Task][Exploration] Attempting to pull the drawer (testing if unlocked) ...")
            try:
                obs, reward, done, info = execute_pull(env, task, drawer='drawer_1', handle='handle_1', loc=drawer_location)
                # If we get here with no exception, skill succeeded and drawer is unlocked!
                print("[Task][Exploration] Drawer was unlocked and opened successfully.")
                drawer_is_unlocked = True
                # Optionally: Push the drawer closed again to restore initial state if needed
                print("[Task][Exploration] Pushing drawer closed to restore state ...")
                obs, reward, done, info = execute_push(env, task, drawer='drawer_1', loc=drawer_location)
            except Exception as e_pull:
                print("[Task][Exploration] Could not pull/open the drawer -- likely locked.")
                print("Exception details:", e_pull)
                drawer_is_unlocked = False
        except Exception as exploration_e:
            print("[Task][Exploration] Exploration phase encountered an error: ", exploration_e)
            drawer_is_unlocked = False

        print(f"[Task][Exploration] Drawer unlocked? {drawer_is_unlocked}")
        print("[Task] === END EXPLORATION PHASE ===")

        # Now, proceed with the rest of the task plan.
        # The remainder of the logic will depend on knowing the state of the drawer (unlocked or locked). 
        # In a full oracle plan, now that we've discovered (drawer-unlocked drawer_1), we proceed to manipulate objects using the available skills.

        # Example plan (must be adapted for your specific target/goal):
        # 1. Go to ready-pose if required
        # 2. Move to the relevant location for the object to pick
        # 3. Pick the object
        # 4. Move to the drawer
        # 5. Open the drawer using handle
        # 6. Place the object inside the drawer
        # 7. Close the drawer
        # The following is a template for filling in such steps.

        # You need to fill in object names and positions specific to your scenario.
        # We'll use placeholder names: 'target_object', 'ready-pose', etc.

        # -- Example plan follows --

        try:
            print("[Task] Moving robot to ready-pose, if required ...")
            if ready_pose is not None:
                curr_loc = positions.get('robot_location', None)
                if curr_loc and curr_loc != ready_pose:
                    obs, reward, done, info = execute_go(env, task, from_loc=curr_loc, to_loc=ready_pose)
            
            # Move to object location
            target_object = 'target_object' if 'target_object' in positions else None
            object_location = positions.get('target_object_location', None)
            if not target_object:
                # Try to guess the object to pick (first available object name that's not drawer or handle)
                for k in positions:
                    if 'object' in k and not 'handle' in k:
                        target_object = k
                        object_location = positions.get(f"{k}_location", None)
                        break
            
            if target_object and object_location:
                print(f"[Task] Moving to {target_object} at {object_location} ...")
                curr_loc = positions.get('robot_location', None)
                if curr_loc != object_location:
                    obs, reward, done, info = execute_go(env, task, from_loc=curr_loc, to_loc=object_location)
                
                # Pick the object
                print(f"[Task] Picking up {target_object} ...")
                obs, reward, done, info = execute_pick(env, task, obj=target_object, loc=object_location)
            
            # Move to drawer location
            print("[Task] Moving to drawer location ...")
            drawer_location = positions.get('drawer_1_location', None)
            curr_loc = positions.get('robot_location', None)
            if drawer_location and curr_loc != drawer_location:
                obs, reward, done, info = execute_go(env, task, from_loc=curr_loc, to_loc=drawer_location)
            
            # If drawer not open, open it using handle
            print("[Task] Opening the drawer by picking handle and pulling ...")
            obs, reward, done, info = execute_pick(env, task, obj='handle_1', loc=drawer_location)
            obs, reward, done, info = execute_pull(env, task, drawer='drawer_1', handle='handle_1', loc=drawer_location)
            
            # Place the object inside the drawer
            print("[Task] Placing the object in the drawer ...")
            obs, reward, done, info = execute_place(env, task, obj=target_object, drawer='drawer_1', loc=drawer_location)
            
            # Push drawer closed
            print("[Task] Closing the drawer ...")
            obs, reward, done, info = execute_push(env, task, drawer='drawer_1', loc=drawer_location)
            
        except Exception as plan_e:
            print("[Task] Error during main plan execution:", plan_e)
            # Additional error handling or fallback as needed

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()