# run_skeleton_task.py (Completed Task with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use externally provided skill primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing the simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (optional)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and States ===
        positions = get_object_positions()
        # The structure of "positions" will depend on your env implementation.
        # Adjust key names below as needed for your environment!
        # For example: positions = {'robot': ..., 'drawer1': ..., 'handle1': ..., 'object1': ...}

        # ---- INITIALIZATION ----
        # Example assignments, adjust as needed for actual key names present in your simulation:
        robot_pos = positions.get('robot', None)
        drawer1_pos = positions.get('drawer1', None)
        handle1_pos = positions.get('handle1', None)
        # If the handle is a separate object: handle1
        # If not, use appropriate names.

        # For exploration, feedback indicates to check for the (drawer-unlocked drawer1) predicate.
        # That is, before we attempt actions like execute_pull, we must ensure that the drawer is unlocked.
        # Because it is not directly observable, we must trigger exploration to check its state.
        # In the provided domain, this is done by attempting execute_pull and observing if it fails,
        # or by some dedicated exploration action, if available. 
        # As per guidelines, we must ONLY use the provided skill primitives (from skill_code).

        # --------- EXPLORATION PHASE ---------
        # (drawer-unlocked drawer1) -- feedback says we may need to discover/ensure this predicate.

        # BASIC STRATEGY:
        # 1. Use the available execute_pull skill and see if it succeeds (i.e., the drawer is unlocked).
        #    If it fails, we conclude the drawer is locked and must choose a different path (not specified here).
        # 2. Use try/except to handle pull failure gracefully.

        # Let's assume we need to:
        #   (1) Move to the handle
        #   (2) Pick the handle (execute_pick_handle)
        #   (3) Attempt to pull the drawer (execute_pull)
        # These skills must be available in skill_code.

        # NOTE: You may need to adjust object/handle names
        handle_name = None
        drawer_name = None
        handle_location = None
        drawer_location = None

        # Identify the handle and drawer by type (based on keys)
        for key in positions.keys():
            if 'handle' in key and handle_name is None:
                handle_name = key
                handle_location = positions[key]
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
                drawer_location = positions[key]
        
        if handle_name is None or drawer_name is None or handle_location is None or drawer_location is None:
            print("[Error] Could not identify handle or drawer in object positions. Aborting task.")
            return

        print("[Exploration] Moving robot to handle location:", handle_location)
        # The move action is likely called execute_go
        # We assume the robot starts elsewhere, so move to the handle's location
        try:
            obs, reward, done = execute_go(
                env, task,
                from_position=robot_pos,   # If robot_pos is the current location as per your simulation
                to_position=handle_location
            )
        except Exception as e:
            print("[Error] Failed to execute 'execute_go' to handle:", e)
            return
        
        print("[Exploration] Picking handle:", handle_name, "at", handle_location)
        try:
            obs, reward, done = execute_pick_handle(
                env, task,
                handle_name, handle_location
            )
        except Exception as e:
            print("[Error] Failed to execute 'execute_pick_handle' on", handle_name, ":", e)
            return
        
        # Attempt to pull the drawer. If it fails, likely the predicate (drawer-unlocked drawer1) is missing.
        print("[Exploration] Attempting to pull the drawer to test if unlocked:", drawer_name)
        try:
            obs, reward, done = execute_pull(
                env, task,
                drawer_name, handle_name, handle_location
            )
            print("[Exploration] Pull succeeded: Drawer is unlocked. Predicate (drawer-unlocked {}) holds.".format(drawer_name))
        except Exception as e:
            # This is where the exploration result is obtained
            print("[Exploration] Pull failed for {}: likely drawer is locked. Missing predicate (drawer-unlocked {})".format(drawer_name, drawer_name))
            # Here you would take additional steps to unlock the drawer if required (not in current skills).
            # For now, halt further plan execution.
            return
        
        # ==== Continue with the Oracle Plan (domain-dependent steps) ====
        # E.g., Place object in drawer, close drawer, etc.
        #
        # Example flow after drawer is opened (add your own oracle plan as needed):
        # Find an object to pick and place in the drawer
        object_name = None
        object_location = None
        for key in positions.keys():
            if key != handle_name and key != drawer_name and 'object' in key:
                object_name = key
                object_location = positions[key]
                break
        
        if object_name is not None and object_location is not None:
            # Move to object
            print("[Task] Moving to object:", object_name, "at", object_location)
            try:
                obs, reward, done = execute_go(
                    env, task,
                    from_position=handle_location,  # robot is at handle after pull
                    to_position=object_location
                )
            except Exception as e:
                print("[Error] Failed to execute 'execute_go' to object:", e)
                return
            
            # Pick the object
            print("[Task] Picking object:", object_name)
            try:
                obs, reward, done = execute_pick(
                    env, task,
                    object_name, object_location
                )
            except Exception as e:
                print("[Error] Failed to execute 'execute_pick' on", object_name, ":", e)
                return
            
            # Move to drawer to place object
            print("[Task] Moving to drawer to place object:", drawer_location)
            try:
                obs, reward, done = execute_go(
                    env, task,
                    from_position=object_location,
                    to_position=drawer_location
                )
            except Exception as e:
                print("[Error] Failed to execute 'execute_go' to drawer for placement:", e)
                return

            # Place the object in the drawer
            print("[Task] Placing object", object_name, "in drawer", drawer_name)
            try:
                obs, reward, done = execute_place(
                    env, task,
                    object_name, drawer_name, drawer_location
                )
            except Exception as e:
                print("[Error] Failed to execute 'execute_place':", e)
                return
            
            # Optionally, push drawer to close it
            print("[Task] Pushing drawer to close it.")
            try:
                obs, reward, done = execute_push(
                    env, task,
                    drawer_name, drawer_location
                )
            except Exception as e:
                print("[Warning] Could not push drawer closed:", e)
        else:
            print("[Task] No object found to pick and place.")
        
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
