# run_skeleton_task.py (Executable code with exploration phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skill implementations directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    env = None
    try:
        # === Environment Setup ===
        env, task = setup_environment()
        descriptions, obs = task.reset()

        # Optional: Video recording setup
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try all exploration skills and observe effects to find missing predicate ===
        # This phase leverages possible exploration actions to check what predicates can be revealed.
        # Use try/except to prevent the whole process from stopping in case an action is not applicable or fails.

        # Assumptions: 'robot', object and location names can be inferred from initial state.
        # Since we lack exact names, iterate over possible combinations using positions and skill set.
        # You may need to adapt 'robot', 'location', and 'object' variables to your environment's actual naming.
        # Here, we use all objects and try skills relevant to exploration.

        # List of predefined exploration action skill names (matching what is likely in skill_code)
        exploration_skills = [
            'execute_go_identify',
            'execute_go_temperature',
            'execute_pick_weight',
            'execute_pick_durability',
            'execute_pull'
        ]
        # Gather objects and locations from object_positions
        objects = []
        locations = []
        for name, pos in positions.items():
            # Heuristic: if name starts with 'obj' or is in object list, it's an object; similarly for locations
            if 'drawer' in name or 'obj' in name or 'handle' in name:
                objects.append(name)
            if 'location' in name or 'pose' in name or 'table' in name:
                locations.append(name)
        # If locations not found, fall back to unique position values
        if not locations:
            locations = list(set([str(t) for t in positions.values()]))

        robot_location = None
        for name in positions:
            if 'robot' in name:
                robot_location = positions[name]
                break

        print("[Exploration] Starting exploration phase to try revealing new predicates...")
        for skill_name in exploration_skills:
            skill_fn = globals().get(skill_name, None)
            if not skill_fn:
                continue
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                if skill_name == 'execute_go_identify' or skill_name == 'execute_go_temperature':
                    if len(locations) >= 2:
                        from_loc = locations[0]
                        to_loc = locations[1]
                        obs, reward, done, info = skill_fn(
                            env,
                            task,
                            robot='robot',
                            from_location=from_loc,
                            to_location=to_loc
                        )
                    else:
                        print("[Exploration] Not enough locations found for go skills.")
                elif skill_name == 'execute_pick_weight' or skill_name == 'execute_pick_durability':
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        obs, reward, done, info = skill_fn(
                            env,
                            task,
                            robot='robot',
                            obj=obj,
                            loc=loc
                        )
                    else:
                        print("[Exploration] Did not find eligible object/location for pick skills.")
                elif skill_name == 'execute_pull':
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        # Assume robot is at correct location and object is present.
                        obs, reward, done, info = skill_fn(
                            env,
                            task,
                            robot='robot',
                            obj=obj,
                            loc=loc
                        )
                    else:
                        print("[Exploration] No eligible object/location for pull.")
            except Exception as e:
                print(f"[Exploration] {skill_name} failed or is not applicable here: {e}")

        print("[Exploration] Completed. Check environment logs or state for new predicate exposure.")

        # === Plan Execution Phase ===
        # After exploration, proceed with regular task-solving using the available predefined skills.
        # In real use, the oracle plan should be loaded here and executed step by step by looking up each skill.

        # Example of generic plan loop using available primitive skills, to be customized for the actual oracle plan.
        oracle_plan = []  # Placeholder for actual oracle plan, which should be filled with plan steps.
        # Example: oracle_plan = [('execute_go', {'from_location': a, 'to_location': b}), ...]

        for action_step in oracle_plan:
            action_name, params = action_step[0], action_step[1]
            skill_fn = globals().get(action_name, None)
            if not skill_fn:
                print(f"[Plan] Skill {action_name} not implemented or not available.")
                continue
            try:
                print(f"[Plan] Executing skill: {action_name} with params {params}")
                obs, reward, done, info = skill_fn(env, task, **params)
                # Optionally: task.get_observation() or any post-effect step here
                if done:
                    print("[Plan] Task complete!")
                    break
            except Exception as e:
                print(f"[Plan] {action_name} failed: {e}")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
