# run_skeleton_task.py (Completed for Exploration Phase to Discover Missing Predicate)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only these, don't define new skill functions.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration phase to find missing predicate.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    missing_predicate_found = False
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase ===

        print("[Exploration] Start searching for missing predicate related to object state...")

        # Since we do not know the missing predicate, we need to run through available skills
        # and see which action triggers a state transition or error. Use only predefined skills.

        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep',
                            'execute_rotate', 'execute_go', 'execute_gripper']

        # Get object/location keys
        objects = [k for k in positions.keys() if 'object' in k or 'obj' in k or 'handle' in k]
        drawers = [k for k in positions.keys() if 'drawer' in k]
        locations = [k for k in positions.keys() if 'loc' in k or 'table' in k or 'pose' in k or 'drawer' in k]

        if len(locations) == 0 and 'ready-pose' not in positions:
            locations = ['ready-pose']

        # We'll try a generic sequence:
        # 1. Try to pick all pickable objects (not handles), possibly at all locations.
        # 2. Try to open all drawers using all handles.
        # 3. Try place, push, sweep, etc.
        # If any action fails due to a missing predicate, or returns an exception, we record it.

        exploration_results = []

        # 1. Try picking up all objects with execute_pick
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    exploration_results.append(('execute_pick', obj, loc, 'success'))
                    print(f"--> Success (execute_pick on {obj} at {loc})")
                except Exception as e:
                    print(f"--> Exception (execute_pick on {obj} at {loc}): {e}")
                    exploration_results.append(('execute_pick', obj, loc, str(e)))
                    # Here, we may infer something about missing predicate (e.g., on-floor, robot-free, need-ready...)

        # 2. Try execute_place for all (obj, drawer, location)
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_place on {obj} into {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        exploration_results.append(('execute_place', obj, drawer, loc, 'success'))
                        print(f"--> Success (execute_place on {obj} in {drawer} at {loc})")
                    except Exception as e:
                        print(f"--> Exception (execute_place on {obj} in {drawer} at {loc}): {e}")
                        exploration_results.append(('execute_place', obj, drawer, loc, str(e)))

        # 3. Try to push every drawer at every location
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    exploration_results.append(('execute_push', drawer, loc, 'success'))
                    print(f"--> Success (execute_push on {drawer} at {loc})")
                except Exception as e:
                    print(f"--> Exception (execute_push on {drawer} at {loc}): {e}")
                    exploration_results.append(('execute_push', drawer, loc, str(e)))

        # 4. Try pulling handles on drawers at every location (execute_pull)
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_pull with {obj} on {drawer} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, obj, loc)
                        exploration_results.append(('execute_pull', obj, drawer, loc, 'success'))
                        print(f"--> Success (execute_pull with {obj} on {drawer} at {loc})")
                    except Exception as e:
                        print(f"--> Exception (execute_pull with {obj} on {drawer} at {loc}): {e}")
                        exploration_results.append(('execute_pull', obj, drawer, loc, str(e)))

        # 5. Try executing sweep and rotate on all objects at all locations
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    exploration_results.append(('execute_sweep', obj, loc, 'success'))
                    print(f"--> Success (execute_sweep on {obj} at {loc})")
                except Exception as e:
                    print(f"--> Exception (execute_sweep on {obj} at {loc}): {e}")
                    exploration_results.append(('execute_sweep', obj, loc, str(e)))

                try:
                    print(f"[Exploration] Trying execute_rotate on {obj} at {loc}")
                    obs, reward, done = execute_rotate(env, task, obj, loc)
                    exploration_results.append(('execute_rotate', obj, loc, 'success'))
                    print(f"--> Success (execute_rotate on {obj} at {loc})")
                except Exception as e:
                    print(f"--> Exception (execute_rotate on {obj} at {loc}): {e}")
                    exploration_results.append(('execute_rotate', obj, loc, str(e)))

        # 6. Try moving the robot to all pairs of locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        exploration_results.append(('execute_go', from_loc, to_loc, 'success'))
                        print(f"--> Success (execute_go from {from_loc} to {to_loc})")
                    except Exception as e:
                        print(f"--> Exception (execute_go from {from_loc} to {to_loc}): {e}")
                        exploration_results.append(('execute_go', from_loc, to_loc, str(e)))

        # 7. Try gripper skill
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            exploration_results.append(('execute_gripper', 'success'))
            print(f"--> Success (execute_gripper)")
        except Exception as e:
            print(f"--> Exception (execute_gripper): {e}")
            exploration_results.append(('execute_gripper', str(e)))

        # === After exploration, analyze results ===
        print("\n[Exploration] Summary of results:")
        for res in exploration_results:
            print(res)
        print("If an action systematically failed due to a missing state/precondition, check the exception messages and domain predicates to infer which predicate is missing in the PDDL problem instance.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()