# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use existing primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Returns a dictionary, e.g., {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase to identify missing predicates ===
        print("[Exploration] Starting exploration to determine missing predicates or unobservable states...")

        # We'll try to sequentially execute available skills and observe for issues, errors, or missing predicates
        # Available skill names:
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]
        
        # Since actual plan/goal is not given, our exploratory strategy is as follows:
        # - For each relevant object (inferred from positions), try each skill,
        # - catch exceptions or failed predicates that reveal missing preconditions/effects,
        # - print/log as we try.

        # Example: Try to pick then place each object. If fails, log the failure.
        # This phase is crucial for debugging plan failure due to missing predicates.

        # We'll start at ready-pose if possible.
        # Use first position as base for exploration.
        obj_names = [name for name in positions.keys()]
        explored_actions = []

        for obj in obj_names:
            # Try to move to the object's location first (if possible)
            obj_pos = positions[obj]

            try:
                print(f"[Exploration] Trying to move to object {obj} at {obj_pos}")
                # Retrieve the robot's current position
                current_pos = positions.get('robot', None)
                if current_pos is not None:
                    # execute_go may require string 'from', 'to' location names, not positions
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_pos,
                        to_location=obj_pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Moved to {obj}'s location.")
                else:
                    print(f"[Exploration] Could not determine robot's current position.")

                # Now try to pick the object with execute_pick
                print(f"[Exploration] Trying to pick {obj}...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=obj,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picking {obj} succeeded.")

                # Try placing the object at a default/reachable location (if any)
                # We'll check if there is a drawer or other placement point
                drawer_candidates = [k for k in obj_names if 'drawer' in k]
                if drawer_candidates:
                    drawer = drawer_candidates[0]
                    drawer_pos = positions[drawer]
                    print(f"[Exploration] Trying to place {obj} in {drawer}...")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        target_object=obj,
                        target_drawer=drawer,
                        target_pos=drawer_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Placing {obj} in {drawer} succeeded.")
                else:
                    print(f"[Exploration] No drawer found for placing {obj}.")
            
            except Exception as e:
                print(f"[Exploration] Action for {obj} failed: {type(e).__name__}: {e}")
                explored_actions.append((obj, str(e)))  # Log failures

        # Further exploration: Try all skills for every object, if appropriate.
        for obj in obj_names:
            obj_pos = positions[obj]
            for skill in available_skills:
                # Skip already tried. Try new skills.
                try:
                    print(f"[Exploration] Attempt skill: {skill} on {obj}")
                    skill_func = globals()[skill]
                    # Compose suitable arguments per skill signature
                    if skill == "execute_pick":
                        obs, reward, done = skill_func(
                            env, task, target_object=obj, target_pos=obj_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                    elif skill == "execute_place":
                        drawer_candidates = [k for k in obj_names if 'drawer' in k]
                        if drawer_candidates:
                            drawer = drawer_candidates[0]
                            drawer_pos = positions[drawer]
                            obs, reward, done = skill_func(
                                env, task, target_object=obj, target_drawer=drawer, target_pos=drawer_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                        else:
                            print("[Exploration] No drawer to place into.")
                            continue
                    elif skill == "execute_go":
                        # Move to this object's location
                        current_pos = positions.get('robot', None)
                        if current_pos is not None:
                            obs, reward, done = skill_func(
                                env, task,
                                from_location=current_pos,
                                to_location=obj_pos,
                                max_steps=100,
                                threshold=0.01,
                                timeout=10.0
                            )
                    elif skill == "execute_pull":
                        obs, reward, done = skill_func(
                            env, task,
                            target_drawer=obj,
                            handle_object=obj,
                            target_pos=obj_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                    elif skill in ["execute_push", "execute_rotate", "execute_sweep", "execute_gripper"]:
                        obs, reward, done = skill_func(env, task)
                    else:
                        print(f"[Exploration] Unrecognized skill {skill}, skipping.")
                        continue
                    print(f"[Exploration] {skill} on {obj} succeeded.")
                except Exception as e:
                    print(f"[Exploration] {skill} on {obj} failed: {type(e).__name__}: {e}")
                    explored_actions.append((f"{skill}({obj})", str(e)))

        print("[Exploration] Completed. Summary of failed actions/potential missing predicates:")
        for fail in explored_actions:
            print(f"  {fail[0]} --> {fail[1]}")

        print("[Exploration] If certain actions fail due to persistent predicate errors, review domain PDDL for missing predicates or misalignments.")

        # === Here you'd proceed to run the actual plan (oracle plan) once exploration/repair is complete and predicates understood. ===
        # You would fill in the exact high-level plan below.

        # Example placeholder for oracle plan execution, only if exploration is passed and plan is available.
        # print("[Task] Running oracle plan steps...")
        # for action_dict in oracle_plan:
        #     try:
        #         # action_dict is, for example, {'skill': 'execute_pick', 'params': {...}}
        #         skill = action_dict['skill']
        #         skill_params = action_dict['params']
        #         obs, reward, done = globals()[skill](env, task, **skill_params)
        #     except Exception as e:
        #         print(f"[Task] Plan step {skill} failed: {e}")
        #         break

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
