# run_skeleton_task.py (Completed Execution Code for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Positions dict: object_name -> (x, y, z)
        positions = get_object_positions()

        # ------------ Exploration Phase for Missing Predicate ------------
        print("=== [Exploration] Checking drawer state predicates ===")
        # Based on feedback: (drawer-open drawer_bottom) is relevant/missing
        # So, identify all drawers and their states to find which predicate is relevant/missing.

        # --- Try all relevant skills to check and trigger drawer-open state ---
        # Assumption: We do not know the status/trigger of 'drawer-open' for 'drawer_bottom'
        #             so we will attempt to interact to reach and verify this state

        # 1. Identify drawers in the scene (simulate; usually by type/name)
        drawer_names = [obj for obj in positions if 'drawer' in obj]
        print("[Exploration] Drawers found:", drawer_names)

        # 2. For each drawer, attempt to open via available skills
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to open: {drawer}")
                # Identify required handle for the drawer (from domain, commonly named handle_x)
                handle_name = None
                for possible_handle in positions.keys():
                    if 'handle' in possible_handle and drawer in possible_handle:
                        handle_name = possible_handle
                        break
                if handle_name is None:
                    # Try generic handle
                    for possible_handle in positions.keys():
                        if 'handle' in possible_handle:
                            handle_name = possible_handle
                            break

                # Get location to move to (use object's position or a known safe location)
                drawer_pos = positions[drawer]
                if handle_name:
                    handle_pos = positions[handle_name]
                else:
                    handle_pos = drawer_pos

                # Get robot's current pose/location
                robot_current = None
                for o in positions:
                    if "robot" in o:
                        robot_current = o
                        break

                if robot_current is None:
                    # Fallback, get location called 'ready-pose' if exists
                    robot_current = 'ready-pose'
                # 3. Move robot to drawer location
                try:
                    print(f"Moving robot to drawer: {drawer} at {drawer_pos}")
                    # execute_go: from, to locations (here, symbolic names not positions)
                    obs, reward, done = execute_go(env, task, from_location=robot_current, to_location=drawer)
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {e}")

                # 4. Pick up the handle if present, else skip
                if handle_name:
                    try:
                        print(f"Picking up handle: {handle_name}")
                        obs, reward, done = execute_pick(env, task, object_name=handle_name, location_name=drawer)
                    except Exception as e:
                        print(f"[Exploration] execute_pick failed: {e}")
                else:
                    print(f"[Exploration] No handle found for {drawer}, skipping handle pick.")

                # 5. Try pulling the drawer (if handle is picked)
                try:
                    print(f"Pulling drawer: {drawer} using handle: {handle_name}")
                    obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle_name, location_name=drawer)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {e}")

                # 6. After attempted pull, check if drawer is open in the observation
                # We assume the task or obs provides the state of predicates such as (drawer-open drawer_bottom)
                obs_info = obs if isinstance(obs, dict) else {}

                is_open = False
                # Look for evidence in obs (depends on task API)
                if 'drawer-open' in obs_info:
                    open_drawers = obs_info['drawer-open']
                    if drawer in open_drawers:
                        is_open = True
                elif f'drawer-open_{drawer}' in obs_info:
                    if obs_info[f'drawer-open_{drawer}']:
                        is_open = True
                else:
                    # fallback: try description feedback or skip
                    pass

                if is_open:
                    print(f">>> Drawer {drawer} is now OPEN (predicate present: (drawer-open {drawer}))")
                else:
                    print(f">>> Drawer {drawer} is NOT open or predicate missing after pull.")

            except Exception as e:
                print(f"[Exploration] Exception while processing {drawer}: {e}")

        # ------------------- End Exploration -------------------

        # ~~~ Oracle Plan/Goal Steps would follow below, using available skills ~~~
        # For this task, main point is exploration and finding the missing predicate.

        print("=== Exploration phase complete. Check above logs for missing/relevant predicates. ===")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
