# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate Identification)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Using only pre-defined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, extended to perform exploration for missing predicates as advised in feedback.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()

        print("[Exploration] Begin predicate exploration to detect missing predicates...")

        # Attempt the available skills in a systematic way to detect possible missing predicates

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # The main idea: try to call every skill and watch if we fail, log/print what predicate might be involved.

        skill_list = [
            "execute_pick", 
            "execute_place", 
            "execute_push", 
            "execute_pull", 
            "execute_sweep", 
            "execute_rotate", 
            "execute_go", 
            "execute_gripper"
        ]
        
        # Try to collect the list of objects, drawers, handles, locations from descriptions/positions if possible.
        # As we don't know the exact structure, attempt basic key extractions.
        object_names = [k for k in positions.keys() if ("object" in k.lower() or "obj" in k.lower())]
        drawer_names = [k for k in positions.keys() if "drawer" in k.lower()]
        location_names = list(set([v for v in positions.values() if type(v)==str]))  # sometimes positions gives room/locations by name

        # For wide coverage, default to using all known named entities for parameter guessing
        # Fallback: If location_names is empty, use all keys
        if not location_names:
            location_names = list(positions.keys())
        
        # For handle names (if they exist)
        handle_names = [k for k in object_names if "handle" in k.lower()]
        non_handle_objects = [k for k in object_names if k not in handle_names]

        # Helper: Try each skill, catch errors indicating missing predicates or type errors
        for skill in skill_list:
            print(f"[Exploration] Trying skill: {skill}")
            try:
                # Prepare dummy parameters based on the skill name (guess from PDDL signatures)
                if skill == "execute_pick":
                    # Try normal object pick (not a handle)
                    if non_handle_objects and location_names:
                        obj = non_handle_objects[0]
                        loc = location_names[0]
                        print(f" - Trying execute_pick({obj}, {loc})")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill == "execute_place":
                    # Place object into a drawer at a location
                    if non_handle_objects and drawer_names and location_names:
                        obj = non_handle_objects[0]
                        drawer = drawer_names[0]
                        loc = location_names[0]
                        print(f" - Trying execute_place({obj}, {drawer}, {loc})")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                elif skill == "execute_push":
                    # Push a drawer
                    if drawer_names and location_names:
                        drawer = drawer_names[0]
                        loc = location_names[0]
                        print(f" - Trying execute_push({drawer}, {loc})")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                elif skill == "execute_pull":
                    # Pull with a handle
                    if drawer_names and handle_names and location_names:
                        drawer = drawer_names[0]
                        handle = handle_names[0]
                        loc = location_names[0]
                        print(f" - Trying execute_pull({drawer}, {handle}, {loc})")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                elif skill == "execute_sweep":
                    # Sweep a non-handle object on floor at location
                    if non_handle_objects and location_names:
                        obj = non_handle_objects[0]
                        loc = location_names[0]
                        print(f" - Trying execute_sweep({obj}, {loc})")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill == "execute_rotate":
                    # Rotate a handle or object at location
                    if handle_names and location_names:
                        obj = handle_names[0]
                    elif non_handle_objects:
                        obj = non_handle_objects[0]
                    else:
                        obj = None
                    if obj and location_names:
                        loc = location_names[0]
                        print(f" - Trying execute_rotate({obj}, {loc})")
                        obs, reward, done = execute_rotate(env, task, obj, loc)
                elif skill == "execute_go":
                    # Move from one location to another
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                    else:
                        from_loc = to_loc = location_names[0]
                    print(f" - Trying execute_go({from_loc}, {to_loc})")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                elif skill == "execute_gripper":
                    print(f" - Trying execute_gripper()")
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f" - No parameter logic defined for skill: {skill}")
                # Log if the skill succeeded
                print(f"   -> Skill {skill} executed (possibly successful, check environment logs).")
            except Exception as e:
                print(f"   [Error] Exception during {skill}: {str(e)}")
                print(f"   [Exploration] This may suggest a missing predicate or unmet precondition related to this action.")
        
        print("[Exploration] Finished trying all skills. Review the output above for any failures/errors that may point to missing predicates (as required by feedback).")
        # Reminder: Plan execution and full goal achievement would be implemented here using proper plan steps and proper parameters,
        # but per instructions, the current focus is on exploration for missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()