# run_skeleton_task.py (Exploration & Missing Predicate Discovery Augmented)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *    # Use only provided skills as given

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation with exploration for missing predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # ------------------------------
        # === Exploration Phase ===
        # ------------------------------
        # We want to discover which predicate causes later steps to fail (e.g., (need-ready)).
        # This is typically handled by checking if actions fail and logging what predicate might be involved.

        # Let's try calling available skills in different ways and watch responses,
        # especially noting if errors or feedback mention need for 'need-ready'.
        # (We do not alter skills or environment logic here!)
        
        skill_names = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # The following phase: try each skill and log errors, focusing on 'need-ready'
        for skill in skill_names:
            skill_fn = globals().get(skill, None)
            if skill_fn is None:
                continue  # skill not implemented, skip
            
            # Prepare dummy/sample arguments based on action kind and positions, just to trigger predicate checks.
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                # This is heuristic: Provide minimal required arguments depending on skill name
                # (Assume keys exist in positions for object names, locations, etc.)
                if skill == 'execute_go':
                    # Needs from and to location
                    locs = list(positions.values())
                    if len(locs) >= 2:
                        obs, reward, done = skill_fn(env, task, from_location=locs[0], to_location=locs[1])
                elif skill == 'execute_pick':
                    # Needs object + location
                    obj_names = list(positions.keys())
                    if obj_names:
                        obj = obj_names[0]
                        pos = positions[obj]
                        obs, reward, done = skill_fn(env, task, obj=obj, location=pos)
                elif skill == 'execute_place':
                    # Needs object, destination, location (simulate using any available)
                    obj_names = list(positions.keys())
                    if len(obj_names) >= 2:
                        obj = obj_names[0]
                        dest = obj_names[1]
                        pos = positions[obj]
                        obs, reward, done = skill_fn(env, task, obj=obj, dest=dest, location=pos)
                elif skill in ['execute_push', 'execute_pull']:
                    # Needs at least object (drawer), handle(h), location
                    obj_names = list(positions.keys())
                    if len(obj_names) >= 2:
                        d = obj_names[0]
                        h = obj_names[1]
                        pos = positions[d]
                        obs, reward, done = skill_fn(env, task, drawer=d, handle=h, location=pos)
                elif skill == 'execute_sweep':
                    # Needs object, location
                    obj_names = list(positions.keys())
                    if obj_names:
                        obj = obj_names[0]
                        pos = positions[obj]
                        obs, reward, done = skill_fn(env, task, obj=obj, location=pos)
                elif skill == 'execute_rotate':
                    # Example: Needs object (simulate with any)
                    obj_names = list(positions.keys())
                    if obj_names:
                        obj = obj_names[0]
                        obs, reward, done = skill_fn(env, task, obj=obj)
                elif skill == 'execute_gripper':
                    obs, reward, done = skill_fn(env, task)
                
                # If skill executes, note it
                print(f"[Exploration] Skill {skill} executed, returned done={done}")

            except Exception as e:
                # Exception string: try to see if '(need-ready)' is referenced
                err_str = str(e)
                print(f"[Exploration] Skill {skill} failed. Exception: {err_str}")
                if "need-ready" in err_str or "need_ready" in err_str or "(need-ready)" in err_str:
                    print("[Exploration] ==> Found that '(need-ready)' is likely missing or required for this skill.")
                continue

        # ----------------------------------------------------------------
        # For full runs, user may now synthesize a plan with correct usage:
        # Plan: After every execute_pick_object or execute_place_object (which sets need-ready), must call execute_go_ready!
        # ----------------------------------------------------------------

        # Example hand-written oracle plan for these actions:
        # 1. execute_pick (object from floor)
        # 2. execute_go_ready (required because (need-ready) set)
        # 3. execute_go (robot moves to, e.g., drawer)
        # 4. execute_place (object in drawer)
        # 5. execute_go_ready (again, (need-ready) set after place)
        # 6. (other skills as appropriate...)

        # Sample Plan Execution
        plan = [
            # ("execute_pick", {...}),
            # ("execute_go_ready", {...}),
            # ("execute_go", {...}),
            # ("execute_place", {...}),
            # ("execute_go_ready", {...}),
        ]

        # Fill in the plan for demonstration (use first object and two arbitrary locations)
        obj_names = list(positions.keys())
        if len(obj_names) >= 2:
            obj = obj_names[0]
            obj2 = obj_names[1]
            # The structure below assumes skill functions use these named parameters; adapt as necessary for your setup.
            plan = [
                ("execute_pick", {"obj": obj, "location": positions[obj]}),
                ("execute_go_ready", {"from_location": positions[obj]}),
                ("execute_go", {"from_location": positions[obj], "to_location": positions[obj2]}),
                ("execute_place", {"obj": obj, "dest": obj2, "location": positions[obj2]}),
                ("execute_go_ready", {"from_location": positions[obj2]}),
            ]

        print("[Oracle Plan] Executing synthesized plan with need-ready linkage...")

        # Step through the plan
        for step in plan:
            skill = step[0]
            kwargs = step[1]
            skill_fn = globals().get(skill, None)
            if skill_fn is None:
                print(f"[Plan] Skill {skill} not implemented, skipping.")
                continue
            try:
                print(f"[Plan] Executing: {skill} with parameters {kwargs}")
                obs, reward, done = skill_fn(env, task, **kwargs)
                if done:
                    print(f"[Plan] Skill {skill} finished task early.")
                    break
            except Exception as e:
                err_str = str(e)
                print(f"[Plan] Skill {skill} failed. Exception: {err_str}")
                if "need-ready" in err_str or "need_ready" in err_str or "(need-ready)" in err_str:
                    print("[Plan] ==> Detected missing (need-ready) predicate requirement. After pick/place, must call execute_go_ready!")
                continue

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
