# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing skills; do not redefine primitives!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.
       This version performs exploratory actions to discover missing predicates as per feedback.
    '''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: {object_name: (x, y, z), ...}
        positions = get_object_positions()
        # If you have names for robot and room, you might retrieve them here.
        # For this exploration, we assume you retrieve all available objects and locations

        # ======== EXPLORATION PHASE TO DISCOVER MISSING PREDICATE =========
        # Based on the exploration domain, try available skills and observe effects.
        # We use only skills in ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, try calling skills with objects and locations from the environment.
        # Note: This is not an oracle plan execution but an exploration per feedback.

        # Find candidate objects and locations
        # For simplicity, assume the keys of positions dict encode type info as e.g. "obj_x", "drawer_y", "loc_z"
        object_names = [name for name in positions if name.startswith('obj_') or name.startswith('object_') or name.startswith('handle')]
        drawer_names = [name for name in positions if name.startswith('drawer_')]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if name.startswith('loc_') or name.startswith('room_') or name == 'ready-pose']
        
        # Fallback if above yields nothing, just get all names to try
        if not location_names:
            location_names = list(positions.keys())
        
        # Initial status
        robot_location = None
        # Attempt to find initial robot location from obs, fallback to a default location name
        try:
            for key in positions:
                if 'robot' in key:
                    robot_location = key
                    break
            if robot_location is None and location_names:
                robot_location = location_names[0]
        except Exception:
            robot_location = location_names[0] if location_names else None

        # Pick a target object for exploration; here just pick the first available
        target_object = object_names[0] if object_names else None
        target_location = location_names[0] if location_names else None
        target_drawer = drawer_names[0] if drawer_names else None
        target_handle = handle_names[0] if handle_names else None

        print("[Exploration] robot_location:", robot_location)
        print("[Exploration] target_object:", target_object)
        print("[Exploration] target_location:", target_location)
        print("[Exploration] target_drawer:", target_drawer)
        print("[Exploration] target_handle:", target_handle)

        # --- 1. Try to execute_move (execute_go) to some location
        try:
            if robot_location and target_location and robot_location != target_location:
                print("[Exploration] Trying execute_go from", robot_location, "to", target_location)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=target_location,
                )
                robot_location = target_location  # Update
        except Exception as e:
            print("[Exploration][Error] execute_go failed:", str(e))

        # --- 2. Try to pick up an object (execute_pick) ---
        try:
            if target_object and target_location:
                print("[Exploration] Trying execute_pick on", target_object, "at", target_location)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=target_object,
                    location_name=target_location
                )
        except Exception as e:
            print("[Exploration][Error] execute_pick failed:", str(e))

        # --- 3. Try to pull on a handle if available (execute_pull) ---
        try:
            if target_drawer and target_handle and robot_location:
                print("[Exploration] Trying execute_pull drawer", target_drawer, "using handle", target_handle, "at", robot_location)
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=target_drawer,
                    handle_name=target_handle,
                    location_name=robot_location
                )
        except Exception as e:
            print("[Exploration][Error] execute_pull failed:", str(e))

        # --- 4. Try to push the drawer closed if opened (execute_push) ---
        try:
            if target_drawer and robot_location:
                print("[Exploration] Trying execute_push drawer", target_drawer, "at", robot_location)
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=target_drawer,
                    location_name=robot_location
                )
        except Exception as e:
            print("[Exploration][Error] execute_push failed:", str(e))

        # --- 5. Try sweep (execute_sweep) if possible ---
        try:
            if target_object and robot_location:
                print("[Exploration] Trying execute_sweep on", target_object, "at", robot_location)
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj_name=target_object,
                    location_name=robot_location
                )
        except Exception as e:
            print("[Exploration][Error] execute_sweep failed:", str(e))

        # --- 6. Try gripper (execute_gripper) if gripper skills exist
        try:
            print("[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
        except Exception as e:
            print("[Exploration][Error] execute_gripper failed:", str(e))

        # At this stage, observe any environmental or task feedback to determine if a new predicate
        # (such as 'lock-known', 'temperature-known', etc.) can be detected or an effect is missing.

        print("===== Exploration Phase Complete. Check logs/outputs or environment for newly observed predicates. =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
