from Object_list import Object
from skill_mapping import RobotController


def Long_horizon_task2_oracle_seq():
    """
    Oracle sequence for: “Place one dice into a drawer.”

    The sequence is written to satisfy the specification and the feedback:
        1) execute_go         (“ready-pose” → “floor”)
        2) execute_pick       (handle_top, floor)
        3) execute_pull       (drawer_top, handle_top, floor)
        4) execute_pick       (dice1, floor)
        5) execute_go_ready   (floor)
        6) execute_place      (dice1, drawer_top, ready-pose)
        7) execute_go_ready   (ready-pose)
        8) execute_push       (drawer_top, ready-pose)

    Mapping between skills and RobotController.execute_movement modes
        execute_go          → mode 7
        execute_pick        → mode 1
        execute_pull        → mode 4
        execute_place       → mode 2
        execute_push        → mode 3
        execute_go_ready    → mode 8

    Axis conventions
        – Handle operations (pick / pull / push) : axis = 0  (approach from x-direction)
        – General object manipulation            : axis = 2  (approach from z-direction)
    """

    robot = RobotController()

    try:
        # ------------------------------------------------------------
        # STEP-1  execute_go  : “ready-pose”  →  “floor”
        # ------------------------------------------------------------
        target_floor_pose = Object.get("floor", None)
        if target_floor_pose is None:          # Fallback – use dice1 pose as a proxy for “floor”
            target_floor_pose = Object["dice1"]
        robot.execute_movement(target_floor_pose, mode=7)

        # ------------------------------------------------------------
        # STEP-2  execute_pick : pick the drawer handle lying on the floor
        # ------------------------------------------------------------
        handle_pose = Object["drawer_top_handle"]
        robot.execute_movement(handle_pose, mode=1, axis=0)

        # ------------------------------------------------------------
        # STEP-3  execute_pull : pull the top drawer open
        # ------------------------------------------------------------
        robot.execute_movement(handle_pose, mode=4, axis=0, distance=0.10)

        # ------------------------------------------------------------
        # STEP-4  execute_pick : pick up dice1 from the floor
        # ------------------------------------------------------------
        dice_pose = Object["dice1"]
        robot.execute_movement(dice_pose, mode=1, axis=2)

        # ------------------------------------------------------------
        # STEP-5  execute_go_ready : robot returns to ready-pose with dice in hand
        # ------------------------------------------------------------
        robot.execute_movement(mode=8)

        # ------------------------------------------------------------
        # STEP-6  execute_place : place dice1 inside the open top drawer
        # ------------------------------------------------------------
        place_pose = Object.get("drawer_top_place_left",
                                Object.get("drawer_top_place_right"))
        robot.execute_movement(place_pose, mode=2, axis=2)

        # ------------------------------------------------------------
        # STEP-7  execute_go_ready : back to ready-pose before closing the drawer
        # ------------------------------------------------------------
        robot.execute_movement(mode=8)

        # ------------------------------------------------------------
        # STEP-8  execute_push : push the drawer closed
        # ------------------------------------------------------------
        robot.execute_movement(handle_pose, mode=3, axis=0, distance=0.10)

    except Exception as e:
        # Basic error reporting – in a real system, consider logging instead
        print(f"[ERROR] Long-horizon task sequence failed: {e}")


def main():
    Long_horizon_task2_oracle_seq()


if __name__ == "__main__":
    main()