# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills; do not redefine any primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Find the Cause of (need-ready) Blocking ===
        #
        # We know from feedback that the missing predicate (need-ready) may block later actions,
        # especially after a "pick" or "place" (since those set need-ready to true).
        # The agent should take "execute_go_ready" to reset need-ready to false when needed.

        # For demonstration, we simulate a simple plan involving picking, placing, and checking blocked states.

        # Dummy naming for skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assume for this scenario there are objects and a drawer present
        # For real usage, use proper names via positions.keys() or observation.

        # For genericity, use placeholders if specific names are not known.
        # Try to infer available objects/locations for demonstration.

        object_names = [name for name in positions if "drawer" not in name and "handle" not in name and "bin" not in name]
        handle_names = [name for name in positions if "handle" in name]
        drawer_names = [name for name in positions if "drawer" in name]
        bin_names = [name for name in positions if "bin" in name]
        location_names = list(set([name for name in positions if "drawer" not in name and "handle" not in name and "bin" not in name]))

        # Dummy fallback:
        if len(object_names) == 0:
            print("[Warning] No manipulable objects in positions dictionary.")
            object_names = list(positions.keys())  # Use anything

        # For demo: just grab first object, first drawer, first handle if available
        obj = object_names[0] if object_names else None
        handle = handle_names[0] if handle_names else None
        drawer = drawer_names[0] if drawer_names else None
        bin_name = bin_names[0] if bin_names else None

        # Use any position as initial robot location
        robot_pos = None
        if positions:
            robot_pos = list(positions.values())[0]

        # --- Example Plan ---
        # 1. Move to the location of an object on the floor ("execute_go").
        # 2. Pick up the object ("execute_pick"). This will set (need-ready).
        # 3. Try to pick a second object without clearing need-ready → should fail.
        # 4. Call "execute_go_ready" (resets need-ready), then proceed.
        # 5. Place the object in the drawer/bin as demo.

        # 1. Move to object
        try:
            print(f"[Task] Moving to object {obj}...")
            obs, reward, done = execute_go(
                env,   # environment
                task,  # task
                from_location='start',  # use correct location name if available
                to_location=obj,  # or location of obj
                max_steps=100,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to move to object {obj}: {e}")

        # 2. Pick the object (this will set need-ready True)
        try:
            print(f"[Task] Picking object {obj}...")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=obj,
                location=obj,  # or location of the obj
                max_steps=100,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to pick object {obj}: {e}")

        # 3. Try to pick *another* object without go_ready: demonstrates (need-ready) blocking.
        # In a real plan, the blocking would be detected by precondition failure or feedback.
        another_obj = object_names[1] if len(object_names) > 1 else None
        if another_obj is not None:
            try:
                print(f"[Task] Attempting to pick another object {another_obj} without 'go_ready'...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=another_obj,
                    location=another_obj,  # location of the other obj
                    max_steps=100,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error/Expected] Could not pick another object without resetting (need-ready): {e}")

        # 4. Take "go_ready" to clear the (need-ready) state.
        try:
            print("[Task] Executing 'go_ready' to reset (need-ready) predicate...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=obj,
                to_location='ready-pose',
                max_steps=100,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Failed to execute go_ready: {e}")
        
        # 5. Now, proceed to place object in bin or drawer
        # For demonstration, check if a bin or drawer is available
        try:
            if bin_name:
                print(f"[Task] Placing object {obj} into bin {bin_name}...")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=obj,
                    location=bin_name,
                    max_steps=100,
                    timeout=10.0
                )
            elif drawer:
                print(f"[Task] Placing object {obj} into drawer {drawer}...")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=obj,
                    location=drawer,
                    max_steps=100,
                    timeout=10.0
                )
            else:
                print("[Warning] No bin or drawer found, skipping place action.")
        except Exception as e:
            print(f"[Error] Failed to place object {obj}: {e}")

        # This demonstrates that after every pick/place/skill that sets (need-ready),
        # you MUST take the corresponding "go_ready" action before doing a new pick/place/etc.

        print("[Exploration] Demonstration completed. (need-ready) predicate confirmed as blocker after pick/place; clearing via 'go_ready' is required.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
