# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (not (need-ready)) is a critical precondition for most actions.
        # We need to explore the environment to determine if the 'need-ready' predicate is missing or incorrectly set.
        # The plan: Try to execute a skill that requires (not (need-ready)) and handle failure if the predicate is missing.

        # For demonstration, we attempt to execute a simple skill (e.g., execute_go) between two locations.
        # If the action fails due to 'need-ready', we will attempt to resolve it by calling execute_go_ready.

        # Find two locations from positions (assuming at least two exist)
        location_names = [name for name, pos in positions.items() if 'location' in name or 'pose' in name or 'drawer' in name or 'bin' in name]
        if len(location_names) < 2:
            # Fallback: just use all keys and pick two
            location_names = list(positions.keys())
        if len(location_names) < 2:
            print("[Exploration] Not enough locations found in object positions for exploration.")
            return

        from_location = location_names[0]
        to_location = location_names[1]

        print(f"[Exploration] Attempting to move from {from_location} to {to_location} using execute_go.")

        try:
            # Try to move the robot using execute_go
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] execute_go succeeded from {from_location} to {to_location}.")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")
            # If failure is due to 'need-ready', try to resolve by calling execute_go_ready
            print("[Exploration] Attempting to resolve 'need-ready' by calling execute_go_ready.")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location
                )
                print("[Exploration] execute_go_ready succeeded. Retrying execute_go.")
                # Retry execute_go after resetting 'need-ready'
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location,
                    to_location
                )
                print(f"[Exploration] execute_go succeeded after resetting 'need-ready'.")
            except Exception as e2:
                print(f"[Exploration] execute_go_ready or second execute_go failed: {e2}")
                print("[Exploration] Unable to resolve missing predicate. Exploration failed.")
                return

        # At this point, we have explored and handled the missing predicate (need-ready).
        # The rest of the plan can proceed as normal, using the available skills.

        # === Example: Try to pick an object if available ===
        object_names = [name for name in positions.keys() if 'object' in name or 'handle' in name]
        if object_names:
            target_object = object_names[0]
            print(f"[Exploration] Attempting to pick {target_object} at {to_location} using execute_pick.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object,
                    to_location
                )
                print(f"[Exploration] execute_pick succeeded for {target_object}.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")
        else:
            print("[Exploration] No objects found to pick.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()