# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (not (need-ready)) is a critical precondition for most actions.
        # We need to explore the environment to determine if 'need-ready' is set and, if so, clear it.

        # 1. Check if the robot is in a 'need-ready' state (i.e., the predicate (need-ready) is true).
        #    If so, we must execute the 'execute_go_ready' skill to clear it.
        #    Since we cannot directly check predicates, we attempt an action that requires (not (need-ready)).
        #    If it fails, we execute 'execute_go_ready' and retry.

        # 2. For demonstration, we attempt to move the robot using 'execute_go'.
        #    If it fails due to 'need-ready', we clear it and try again.

        # 3. This logic can be reused before any action that requires (not (need-ready)).

        # === Example: Try to move the robot to a new location ===
        # Find two distinct locations from positions
        location_names = [name for name, pos in positions.items() if 'location' in name or 'pose' in name or 'drawer' in name or 'bin' in name]
        if len(location_names) < 2:
            # Fallback: Use any two keys
            location_names = list(positions.keys())[:2]
        if len(location_names) < 2:
            print("[Exploration] Not enough locations to perform exploration.")
            return

        from_location = location_names[0]
        to_location = location_names[1]

        print(f"[Exploration] Attempting to move from {from_location} to {to_location}.")

        # Try to execute 'execute_go' (move) skill
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] Successfully moved from {from_location} to {to_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to move: {e}")
            print("[Exploration] Attempting to clear 'need-ready' state by executing 'execute_go_ready'.")

            # Try to clear 'need-ready' by executing 'execute_go_ready'
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location
                )
                print("[Exploration] Executed 'execute_go_ready' to clear 'need-ready'.")

                # Retry the move
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location,
                    to_location
                )
                print(f"[Exploration] Successfully moved from {from_location} to {to_location} after clearing 'need-ready'.")
            except Exception as e2:
                print(f"[Exploration] Still failed to move after clearing 'need-ready': {e2}")
                print("[Exploration] Exploration failed. There may be another missing predicate or issue.")
                return

        # === Exploration Complete ===
        print("[Exploration] Exploration phase complete. The missing predicate was likely 'need-ready'.")
        print("[Exploration] You must ensure (not (need-ready)) before executing most skills.")

        # === Example: Continue with further actions as needed ===
        # For demonstration, try to pick an object if available
        object_names = [name for name in positions.keys() if 'object' in name or 'handle' in name]
        if object_names:
            obj_name = object_names[0]
            print(f"[Exploration] Attempting to pick object: {obj_name} at {to_location}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name,
                    to_location
                )
                print(f"[Exploration] Successfully picked {obj_name}.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj_name}: {e}")
                print("[Exploration] If this fails due to 'need-ready', execute 'execute_go_ready' and retry.")
                try:
                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        to_location
                    )
                    print("[Exploration] Executed 'execute_go_ready' before picking.")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name,
                        to_location
                    )
                    print(f"[Exploration] Successfully picked {obj_name} after clearing 'need-ready'.")
                except Exception as e2:
                    print(f"[Exploration] Still failed to pick {obj_name}: {e2}")
        else:
            print("[Exploration] No objects found to pick.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
