# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (not (need-ready)) ===
        # The feedback indicates that (not (need-ready)) is a missing predicate in the state.
        # Many actions in the domain require (not (need-ready)) as a precondition.
        # If (need-ready) is True, we must execute the ready action to clear it.

        # We'll define a helper to check if (need-ready) is set in the current observation.
        def is_need_ready(obs):
            # Try to detect the 'need-ready' predicate in the observation/state
            # This depends on the structure of obs; fallback to False if not found
            try:
                if 'need-ready' in obs and obs['need-ready']:
                    return True
                if 'predicates' in obs and 'need-ready' in obs['predicates']:
                    return obs['predicates']['need-ready']
                # If obs is a list of predicates
                if isinstance(obs, list):
                    for pred in obs:
                        if isinstance(pred, str) and pred.strip() == '(need-ready)':
                            return True
                # If obs is a dict of booleans
                if isinstance(obs, dict):
                    if obs.get('need-ready', False):
                        return True
            except Exception:
                pass
            return False

        # === Main Plan Execution ===
        # For demonstration, we will perform a simple exploration loop to clear (need-ready)
        # and then attempt a generic skill sequence using only available skills.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll attempt to clear (need-ready) if present, then perform a simple pick and place if possible.

        max_steps = 20
        step_count = 0
        done = False

        while step_count < max_steps and not done:
            obs = task.get_observation()
            # 1. If (need-ready) is set, execute_go_ready to clear it
            if is_need_ready(obs):
                # Find current robot location
                robot_pos = None
                if 'robot-at' in obs:
                    robot_pos = obs['robot-at']
                elif 'robot_position' in obs:
                    robot_pos = obs['robot_position']
                elif 'predicates' in obs and 'robot-at' in obs['predicates']:
                    robot_pos = obs['predicates']['robot-at']
                else:
                    # Try to infer from positions
                    robot_pos = positions.get('robot', None)
                if robot_pos is None:
                    print("[Exploration] Could not determine robot position for go_ready.")
                    break
                try:
                    # execute_go_ready(from_location)
                    print("[Exploration] Executing execute_go_ready to clear (need-ready).")
                    obs, reward, done = execute_go_ready(env, task, from_location=robot_pos)
                except Exception as e:
                    print("[Exploration] Exception during execute_go_ready:", e)
                    break
                step_count += 1
                continue

            # 2. Try to pick an object from the floor if hand is empty and robot is free
            # Find an object on the floor
            pickable_obj = None
            pickable_pos = None
            for obj_name, pos in positions.items():
                if obj_name == 'robot':
                    continue
                # Check if object is on the floor (using obs or positions)
                is_on_floor = False
                if 'on-floor' in obs:
                    if obj_name in obs['on-floor']:
                        is_on_floor = True
                elif 'predicates' in obs and 'on-floor' in obs['predicates']:
                    if obj_name in obs['predicates']['on-floor']:
                        is_on_floor = True
                # Fallback: assume all objects except robot are on floor
                if is_on_floor or True:
                    pickable_obj = obj_name
                    pickable_pos = pos
                    break
            if pickable_obj is not None:
                try:
                    print(f"[Exploration] Attempting to pick {pickable_obj} at {pickable_pos}")
                    obs, reward, done = execute_pick(env, task, obj=pickable_obj, location=pickable_pos)
                except Exception as e:
                    print("[Exploration] Exception during execute_pick:", e)
                step_count += 1
                continue

            # 3. Try to place the object if holding one
            holding_obj = None
            if 'holding' in obs:
                holding_obj = obs['holding']
            elif 'predicates' in obs and 'holding' in obs['predicates']:
                holding_obj = obs['predicates']['holding']
            if holding_obj:
                # Try to place to a bin or drawer if available
                place_location = None
                for obj_name in positions:
                    if 'bin' in obj_name or 'drawer' in obj_name:
                        place_location = positions[obj_name]
                        break
                if place_location is not None:
                    try:
                        print(f"[Exploration] Attempting to place {holding_obj} to {place_location}")
                        obs, reward, done = execute_place(env, task, obj=holding_obj, location=place_location)
                    except Exception as e:
                        print("[Exploration] Exception during execute_place:", e)
                    step_count += 1
                    continue

            # 4. Try to move to a new location if possible
            # Find a new location to go to
            current_loc = None
            if 'robot-at' in obs:
                current_loc = obs['robot-at']
            elif 'robot_position' in obs:
                current_loc = obs['robot_position']
            elif 'predicates' in obs and 'robot-at' in obs['predicates']:
                current_loc = obs['predicates']['robot-at']
            # Find a different location
            target_loc = None
            for loc_name in positions:
                if loc_name != current_loc and loc_name != 'robot':
                    target_loc = positions[loc_name]
                    break
            if current_loc is not None and target_loc is not None:
                try:
                    print(f"[Exploration] Moving from {current_loc} to {target_loc}")
                    obs, reward, done = execute_go(env, task, from_location=current_loc, to_location=target_loc)
                except Exception as e:
                    print("[Exploration] Exception during execute_go:", e)
                step_count += 1
                continue

            # 5. If nothing else, try a no-op skill
            try:
                print("[Exploration] Executing execute_gripper as fallback.")
                obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print("[Exploration] Exception during execute_gripper:", e)
            step_count += 1

        print("===== Exploration Phase Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
