# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that blocks some actions.
        # We will attempt to execute actions and observe if (need-ready) blocks progress.
        # If so, we will use the available skill 'execute_go_ready' to clear (need-ready).

        # For demonstration, let's try to pick up an object, then immediately try to pick another.
        # If the second pick fails due to (need-ready), we will call execute_go_ready.

        # Find an object on the floor and its location
        object_on_floor = None
        object_on_floor_pos = None
        for obj_name, pos in positions.items():
            if 'floor' in obj_name or 'obj' in obj_name or 'object' in obj_name:
                object_on_floor = obj_name
                object_on_floor_pos = pos
                break

        # Find a drawer and its location
        drawer_name = None
        drawer_pos = None
        for obj_name, pos in positions.items():
            if 'drawer' in obj_name:
                drawer_name = obj_name
                drawer_pos = pos
                break

        # Find a handle and its location
        handle_name = None
        handle_pos = None
        for obj_name, pos in positions.items():
            if 'handle' in obj_name:
                handle_name = obj_name
                handle_pos = pos
                break

        # Find a bin and its location (if any)
        bin_name = None
        bin_pos = None
        for obj_name, pos in positions.items():
            if 'bin' in obj_name:
                bin_name = obj_name
                bin_pos = pos
                break

        # Find the robot's current location
        robot_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name or 'ready-pose' in obj_name:
                robot_location = obj_name
                break

        # If not found, use a default
        if robot_location is None:
            robot_location = 'ready-pose'

        # === Example Plan Execution with Exploration for (need-ready) ===

        # 1. Move to the object on the floor (if not already there)
        try:
            if object_on_floor_pos is not None and robot_location != object_on_floor_pos:
                print(f"[Task] Moving to {object_on_floor_pos} to pick up {object_on_floor}")
                obs, reward, done = execute_go(env, task, robot_location, object_on_floor_pos)
                robot_location = object_on_floor_pos
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")

        # 2. Try to pick up the object from the floor
        try:
            print(f"[Task] Attempting to pick up {object_on_floor} at {object_on_floor_pos}")
            obs, reward, done = execute_pick(env, task, object_on_floor, object_on_floor_pos)
        except Exception as e:
            print(f"[Error] Failed to pick up object: {e}")

        # 3. Try to pick up another object (simulate repeated pick to trigger (need-ready))
        #    If (need-ready) blocks, call execute_go_ready to clear it.
        try:
            print(f"[Task] Attempting to pick up {object_on_floor} again to test (need-ready) blocking")
            obs, reward, done = execute_pick(env, task, object_on_floor, object_on_floor_pos)
        except Exception as e:
            print(f"[Info] Action blocked, likely due to (need-ready): {e}")
            print("[Task] Executing execute_go_ready to clear (need-ready)")
            try:
                obs, reward, done = execute_go_ready(env, task, robot_location)
                print("[Task] (need-ready) cleared, can proceed with next actions.")
            except Exception as e2:
                print(f"[Error] Failed to execute execute_go_ready: {e2}")

        # 4. Continue with further actions as needed (e.g., place object, open drawer, etc.)
        #    For demonstration, try to place the object into a bin or drawer if available.
        try:
            if bin_name is not None and bin_pos is not None:
                print(f"[Task] Placing {object_on_floor} into bin at {bin_pos}")
                obs, reward, done = execute_place(env, task, object_on_floor, bin_pos)
            elif drawer_name is not None and drawer_pos is not None:
                print(f"[Task] Placing {object_on_floor} into drawer {drawer_name} at {drawer_pos}")
                obs, reward, done = execute_place(env, task, object_on_floor, drawer_name, drawer_pos)
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")

        # 5. If (need-ready) is set again, clear it
        try:
            print("[Task] Checking if (need-ready) needs to be cleared again.")
            obs, reward, done = execute_go_ready(env, task, robot_location)
        except Exception as e:
            print(f"[Info] (need-ready) may not be set, skipping execute_go_ready: {e}")

        print("[Task] Exploration phase complete. If (need-ready) was blocking, it should now be cleared.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
