# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must call the skill that clears it.
        # The action that clears (need-ready) is execute_go_ready.

        # For demonstration, we will perform a minimal exploration:
        # 1. Try to pick an object (if available) from the floor.
        # 2. If (need-ready) is set (i.e., further actions are blocked), call execute_go_ready to clear it.
        # 3. Continue with the next action.

        # === Example: Find an object on the floor and pick it up ===
        # We'll use the positions dictionary to find an object on the floor.
        # The actual object names and positions depend on your environment setup.

        # Find an object on the floor (excluding handles)
        object_to_pick = None
        object_pos = None
        for obj_name, pos in positions.items():
            if 'handle' in obj_name:
                continue
            # Assume positions dict has info about whether the object is on the floor
            if pos.get('on_floor', False):
                object_to_pick = obj_name
                object_pos = pos
                break

        if object_to_pick is not None:
            print(f"[Exploration] Attempting to pick object: {object_to_pick} at {object_pos}")

            # 1. Move to the object's location if not already there
            robot_pos = positions.get('robot', {}).get('location', None)
            obj_location = object_pos.get('location', None)
            if robot_pos is not None and obj_location is not None and robot_pos != obj_location:
                try:
                    print(f"[Exploration] Moving robot from {robot_pos} to {obj_location}")
                    obs, reward, done, info = execute_go(env, task, robot_pos, obj_location)
                except Exception as e:
                    print(f"[Error] Failed to move robot: {e}")

            # 2. Try to pick the object
            try:
                obs, reward, done, info = execute_pick(env, task, object_to_pick, obj_location)
                print(f"[Exploration] Picked object: {object_to_pick}")
            except Exception as e:
                print(f"[Error] Failed to pick object: {e}")

            # 3. After picking, (need-ready) is likely set. Try to perform another action (e.g., place), which should fail if (need-ready) is not cleared.
            # So, we must call execute_go_ready to clear (need-ready).
            try:
                print(f"[Exploration] Calling execute_go_ready to clear (need-ready)")
                obs, reward, done, info = execute_go_ready(env, task, obj_location)
            except Exception as e:
                print(f"[Error] Failed to execute go_ready: {e}")

            # 4. Now, try to place the object into a bin or drawer if available
            # For demonstration, try to place into a bin if 'bin' exists in positions
            if 'bin' in positions:
                bin_location = positions['bin'].get('location', None)
                if bin_location is not None:
                    try:
                        print(f"[Exploration] Moving to bin at {bin_location}")
                        obs, reward, done, info = execute_go(env, task, obj_location, bin_location)
                    except Exception as e:
                        print(f"[Error] Failed to move to bin: {e}")
                    try:
                        print(f"[Exploration] Placing object {object_to_pick} into bin")
                        obs, reward, done, info = execute_place(env, task, object_to_pick, bin_location)
                    except Exception as e:
                        print(f"[Error] Failed to place object into bin: {e}")
            else:
                print("[Exploration] No bin found in positions; skipping place step.")

        else:
            print("[Exploration] No suitable object found on the floor to pick.")

        # === End of Exploration Phase ===

        # You can expand the above logic to cover more actions and objects as needed.
        # The key is: after any action that sets (need-ready), you must call execute_go_ready to clear it before proceeding.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()