# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase for Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that blocks further actions.
        # We must explore the environment to discover which action or sequence resolves (need-ready).
        # According to the domain, execute_go_ready removes (need-ready).
        # However, since only predefined skills are allowed, and 'execute_go_ready' is not in the available skills,
        # we must use the available skills to explore and identify the missing predicate.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # The plan: Try to execute each available skill and observe if (need-ready) is removed or persists.
        # We will print the state after each action to check for changes.

        # For demonstration, we will attempt to use each skill in a safe, generic way.
        # In a real scenario, you would use the actual oracle plan and object/parameter info.

        # Example: Try to move the robot (execute_go) between two locations if possible.
        location_keys = [k for k, v in positions.items() if v.get('type', '') == 'location']
        object_keys = [k for k, v in positions.items() if v.get('type', '') == 'object']
        drawer_keys = [k for k, v in positions.items() if v.get('type', '') == 'drawer']
        handle_keys = [k for k, v in positions.items() if v.get('is_handle', False)]

        # Fallbacks in case the above info is not available
        if len(location_keys) < 2:
            location_keys = list(positions.keys())[:2]
        if len(object_keys) == 0:
            object_keys = [k for k in positions.keys() if 'obj' in k or 'object' in k]
        if len(drawer_keys) == 0:
            drawer_keys = [k for k in positions.keys() if 'drawer' in k]
        if len(handle_keys) == 0:
            handle_keys = [k for k in positions.keys() if 'handle' in k]

        # Print initial state for debugging
        print("[Exploration] Initial object positions:", positions)
        print("[Exploration] Locations:", location_keys)
        print("[Exploration] Objects:", object_keys)
        print("[Exploration] Drawers:", drawer_keys)
        print("[Exploration] Handles:", handle_keys)

        # Try to execute each skill and observe the effect
        # 1. Try execute_go (move between two locations)
        if len(location_keys) >= 2:
            from_loc = location_keys[0]
            to_loc = location_keys[1]
            try:
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done, info = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] execute_go result:", obs, reward, done, info)
            except Exception as e:
                print("[Exploration] execute_go failed:", e)

        # 2. Try execute_pick (pick up an object)
        if len(object_keys) > 0 and len(location_keys) > 0:
            obj = object_keys[0]
            loc = location_keys[0]
            try:
                print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                obs, reward, done, info = execute_pick(env, task, obj, loc)
                print("[Exploration] execute_pick result:", obs, reward, done, info)
            except Exception as e:
                print("[Exploration] execute_pick failed:", e)

        # 3. Try execute_place (place object into a drawer or bin)
        if len(object_keys) > 0 and len(drawer_keys) > 0 and len(location_keys) > 0:
            obj = object_keys[0]
            drawer = drawer_keys[0]
            loc = location_keys[0]
            try:
                print(f"[Exploration] Trying execute_place on {obj} into {drawer} at {loc}")
                obs, reward, done, info = execute_place(env, task, obj, drawer, loc)
                print("[Exploration] execute_place result:", obs, reward, done, info)
            except Exception as e:
                print("[Exploration] execute_place failed:", e)

        # 4. Try execute_push (push a drawer closed)
        if len(drawer_keys) > 0 and len(location_keys) > 0:
            drawer = drawer_keys[0]
            loc = location_keys[0]
            try:
                print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                obs, reward, done, info = execute_push(env, task, drawer, loc)
                print("[Exploration] execute_push result:", obs, reward, done, info)
            except Exception as e:
                print("[Exploration] execute_push failed:", e)

        # 5. Try execute_pull (pull a drawer open using a handle)
        if len(drawer_keys) > 0 and len(handle_keys) > 0 and len(location_keys) > 0:
            drawer = drawer_keys[0]
            handle = handle_keys[0]
            loc = location_keys[0]
            try:
                print(f"[Exploration] Trying execute_pull on {drawer} with {handle} at {loc}")
                obs, reward, done, info = execute_pull(env, task, drawer, handle, loc)
                print("[Exploration] execute_pull result:", obs, reward, done, info)
            except Exception as e:
                print("[Exploration] execute_pull failed:", e)

        # 6. Try execute_sweep (no-op)
        if len(object_keys) > 0 and len(location_keys) > 0:
            obj = object_keys[0]
            loc = location_keys[0]
            try:
                print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                obs, reward, done, info = execute_sweep(env, task, obj, loc)
                print("[Exploration] execute_sweep result:", obs, reward, done, info)
            except Exception as e:
                print("[Exploration] execute_sweep failed:", e)

        # 7. Try execute_gripper (no-op)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done, info = execute_gripper(env, task)
            print("[Exploration] execute_gripper result:", obs, reward, done, info)
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        # After each action, check if the predicate (need-ready) is still present in the state.
        # Since we do not have direct access to the PDDL state, we rely on the environment's feedback or info.
        # If the environment provides a way to query predicates, do so here.
        # Otherwise, observe if actions are blocked by (need-ready) and print relevant info.

        print("[Exploration] If actions are blocked by (need-ready), consider that execute_go_ready is required to clear it.")
        print("[Exploration] Since execute_go_ready is not in available skills, the missing predicate is (need-ready).")
        print("[Exploration] Exploration complete. Please update the domain or available skills to include execute_go_ready if needed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
