# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicate (need-ready).'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that (need-ready) is a missing predicate that blocks further actions.
        # We must explore the environment to determine when (need-ready) is set and cleared.
        # In the domain, (need-ready) is set as an effect of pick/place actions, and cleared by execute_go_ready.
        # Therefore, after any pick/place, we must call execute_go_ready before further actions.

        # === Example Plan Execution with Exploration ===
        # For demonstration, let's assume a generic plan:
        # 1. Move to object location
        # 2. Pick object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to destination (e.g., drawer/bin)
        # 5. Place object (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. (Repeat as needed)

        # For this example, we will use placeholder object and location names.
        # Replace 'object_1', 'drawer_1', 'bin', 'location_a', etc. with actual names from your environment.

        # Example object and location names (replace with actuals)
        object_name = None
        object_pos = None
        for k in positions:
            if 'object' in k or 'obj' in k:
                object_name = k
                object_pos = positions[k]
                break
        if object_name is None:
            print("[Exploration] No object found in positions. Exiting.")
            return

        # Find a drawer and its position
        drawer_name = None
        drawer_pos = None
        for k in positions:
            if 'drawer' in k:
                drawer_name = k
                drawer_pos = positions[k]
                break
        if drawer_name is None:
            print("[Exploration] No drawer found in positions. Exiting.")
            return

        # Find a bin and its position (if available)
        bin_name = None
        bin_pos = None
        for k in positions:
            if 'bin' in k:
                bin_name = k
                bin_pos = positions[k]
                break

        # Find robot's initial location
        robot_location = None
        for k in positions:
            if 'robot' in k or 'ready' in k:
                robot_location = k
                break
        if robot_location is None:
            robot_location = 'ready-pose'  # fallback

        # Step 1: Move to object location (if not already there)
        try:
            print(f"[Task] Moving to object {object_name} at {object_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=object_name + "_location" if object_name + "_location" in positions else object_pos
            )
            robot_location = object_name + "_location" if object_name + "_location" in positions else object_pos
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")

        # Step 2: Pick the object (sets need-ready)
        try:
            print(f"[Task] Picking up {object_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")

        # Step 3: Go to ready pose (clears need-ready)
        try:
            print("[Task] Going to ready pose to clear need-ready")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location
            )
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")

        # Step 4: Move to drawer location
        try:
            print(f"[Task] Moving to drawer {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_name + "_location" if drawer_name + "_location" in positions else drawer_pos
            )
            robot_location = drawer_name + "_location" if drawer_name + "_location" in positions else drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")

        # Step 5: Place object into drawer (sets need-ready)
        try:
            print(f"[Task] Placing {object_name} into {drawer_name}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_name,
                drawer_name=drawer_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")

        # Step 6: Go to ready pose (clears need-ready)
        try:
            print("[Task] Going to ready pose to clear need-ready after place")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=robot_location
            )
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after place: {e}")

        # === Exploration Summary ===
        print("[Exploration] If at any point an action fails due to (need-ready), ensure execute_go_ready is called before retrying.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
