# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the exploration domain knowledge to try to discover which predicate is missing.
        # This is done by attempting to execute available skills and observing which preconditions fail.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we attempt to execute each skill and catch any errors
        # that may indicate a missing predicate or precondition.
        # In a real system, you would use the plan or feedback to focus on likely-missing predicates.

        # We'll log which skills fail due to missing predicates/preconditions.
        failed_skills = []
        for skill_name in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                # Prepare dummy/default arguments for each skill
                # (In practice, you would use actual objects/locations from the environment)
                if skill_name == 'execute_go':
                    # Try moving from current robot position to another location
                    robot_pos = None
                    for k in positions:
                        if 'robot' in k or 'ready' in k:
                            robot_pos = k
                            break
                    # Find a different location
                    to_pos = None
                    for k in positions:
                        if k != robot_pos:
                            to_pos = k
                            break
                    if robot_pos and to_pos:
                        obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=to_pos)
                elif skill_name == 'execute_pick':
                    # Try picking up an object from the floor
                    obj = None
                    for k in positions:
                        if 'obj' in k or 'object' in k:
                            obj = k
                            break
                    loc = None
                    for k in positions:
                        if 'loc' in k or 'location' in k or 'room' in k:
                            loc = k
                            break
                    if obj and loc:
                        obs, reward, done = execute_pick(env, task, obj=obj, location=loc)
                elif skill_name == 'execute_place':
                    # Try placing an object into a drawer or bin
                    obj = None
                    for k in positions:
                        if 'obj' in k or 'object' in k:
                            obj = k
                            break
                    drawer = None
                    for k in positions:
                        if 'drawer' in k:
                            drawer = k
                            break
                    loc = None
                    for k in positions:
                        if 'loc' in k or 'location' in k or 'room' in k:
                            loc = k
                            break
                    if obj and drawer and loc:
                        obs, reward, done = execute_place(env, task, obj=obj, drawer=drawer, location=loc)
                elif skill_name == 'execute_push':
                    # Try pushing a drawer closed
                    drawer = None
                    for k in positions:
                        if 'drawer' in k:
                            drawer = k
                            break
                    loc = None
                    for k in positions:
                        if 'loc' in k or 'location' in k or 'room' in k:
                            loc = k
                            break
                    if drawer and loc:
                        obs, reward, done = execute_push(env, task, drawer=drawer, location=loc)
                elif skill_name == 'execute_pull':
                    # Try pulling a drawer open using a handle
                    drawer = None
                    handle = None
                    for k in positions:
                        if 'drawer' in k:
                            drawer = k
                        if 'handle' in k:
                            handle = k
                    loc = None
                    for k in positions:
                        if 'loc' in k or 'location' in k or 'room' in k:
                            loc = k
                            break
                    if drawer and handle and loc:
                        obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=loc)
                elif skill_name == 'execute_sweep':
                    # Try sweeping an object on the floor
                    obj = None
                    for k in positions:
                        if 'obj' in k or 'object' in k:
                            obj = k
                            break
                    loc = None
                    for k in positions:
                        if 'loc' in k or 'location' in k or 'room' in k:
                            loc = k
                            break
                    if obj and loc:
                        obs, reward, done = execute_sweep(env, task, obj=obj, location=loc)
                elif skill_name == 'execute_rotate':
                    # Try rotating an object (if supported)
                    obj = None
                    for k in positions:
                        if 'obj' in k or 'object' in k:
                            obj = k
                            break
                    if obj:
                        obs, reward, done = execute_rotate(env, task, obj=obj)
                elif skill_name == 'execute_gripper':
                    # Try gripper action (no-op)
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Unknown skill: {skill_name}")
            except Exception as e:
                print(f"[Exploration] Skill {skill_name} failed: {e}")
                failed_skills.append((skill_name, str(e)))

        # After exploration, print summary of failed skills
        if failed_skills:
            print("[Exploration] The following skills failed (possible missing predicates or preconditions):")
            for skill, err in failed_skills:
                print(f"  - {skill}: {err}")
        else:
            print("[Exploration] All skills executed without error (unexpected).")

        # === (Optional) Main Task Plan Execution ===
        # After identifying missing predicates, you would normally proceed to execute the oracle plan.
        # For this skeleton, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()