# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to identify the missing predicate.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe any unexpected behavior or missing state.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers to see if any action fails due to missing predicate
        # For demonstration, we will attempt to pick, pull, and place objects, and observe for errors

        # Get all objects and drawers from positions (assuming keys are object names)
        object_names = list(positions.keys())
        # For this example, we assume drawers are named with 'drawer' in their name
        drawer_names = [name for name in object_names if 'drawer' in name]
        non_drawer_objects = [name for name in object_names if name not in drawer_names]

        # Assume we have a list of locations (from positions or a predefined list)
        location_names = set()
        for pos in positions.values():
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
        if not location_names:
            # Fallback: use some default locations
            location_names = {'table', 'floor', 'bin', 'ready-pose'}

        # Try to pick each object from the floor (if possible)
        for obj in non_drawer_objects:
            try:
                obj_info = positions[obj]
                # Try to get location
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    obj_loc = obj_info['location']
                else:
                    obj_loc = 'floor'  # fallback
                print(f"[Exploration] Attempting to pick object '{obj}' at location '{obj_loc}'")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    obj_loc
                )
                if done:
                    print(f"[Exploration] Successfully picked '{obj}'")
                else:
                    print(f"[Exploration] Could not pick '{obj}' (possibly due to missing predicate)")
            except Exception as e:
                print(f"[Exploration] Exception during pick of '{obj}': {e}")

        # Try to pull each drawer (if possible)
        for drawer in drawer_names:
            try:
                drawer_info = positions[drawer]
                # Try to get location
                if isinstance(drawer_info, dict) and 'location' in drawer_info:
                    drawer_loc = drawer_info['location']
                else:
                    drawer_loc = 'table'  # fallback
                # Try to find a handle for this drawer
                handle_name = None
                for obj in non_drawer_objects:
                    if 'handle' in obj and drawer in obj:
                        handle_name = obj
                        break
                if handle_name is None:
                    # Try to find any handle
                    for obj in non_drawer_objects:
                        if 'handle' in obj:
                            handle_name = obj
                            break
                if handle_name is None:
                    print(f"[Exploration] No handle found for drawer '{drawer}', skipping pull.")
                    continue
                print(f"[Exploration] Attempting to pull drawer '{drawer}' with handle '{handle_name}' at location '{drawer_loc}'")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_name,
                    drawer_loc
                )
                if done:
                    print(f"[Exploration] Successfully pulled '{drawer}'")
                else:
                    print(f"[Exploration] Could not pull '{drawer}' (possibly due to missing predicate)")
            except Exception as e:
                print(f"[Exploration] Exception during pull of '{drawer}': {e}")

        # Try to place each object into a bin or drawer (if possible)
        for obj in non_drawer_objects:
            try:
                obj_info = positions[obj]
                # Try to get location
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    obj_loc = obj_info['location']
                else:
                    obj_loc = 'floor'  # fallback
                # Try to place into bin
                print(f"[Exploration] Attempting to place object '{obj}' into bin at location 'bin'")
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj,
                    'bin',
                    obj_loc
                )
                if done:
                    print(f"[Exploration] Successfully placed '{obj}' into bin")
                else:
                    print(f"[Exploration] Could not place '{obj}' into bin (possibly due to missing predicate)")
            except Exception as e:
                print(f"[Exploration] Exception during place of '{obj}': {e}")

        # After exploration, print a summary
        print("[Exploration] Exploration phase complete. Check above logs for any actions that failed due to missing predicates.")
        print("[Exploration] If certain actions consistently fail, the corresponding predicate (e.g., lock-known, identified, etc.) may be missing in the initial state or domain.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # At this point, you would normally execute the oracle plan step-by-step using the available skills.
        # For this code, the focus is on the exploration phase to identify missing predicates as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
