# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that may block further actions.
        # We must ensure that after actions that set (need-ready), we perform the necessary skill to clear it.
        # According to the domain, execute_go_ready clears (need-ready).
        # We will attempt to perform execute_go_ready whenever (need-ready) is set.

        # For demonstration, let's assume a generic plan:
        # 1. Pick an object from the floor (which sets need-ready)
        # 2. Go to ready pose (to clear need-ready)
        # 3. Move to a drawer
        # 4. Place the object into the drawer (which sets need-ready)
        # 5. Go to ready pose (to clear need-ready)
        # 6. (Continue as needed...)

        # For this example, we will use the first object and first drawer found in positions.
        # The actual plan should be generated by a planner, but here we focus on the missing predicate handling.

        # Find an object on the floor and a drawer
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        ready_pose = None

        # Identify objects and locations
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif 'object' in name and object_name is None:
                object_name = name
                object_pos = pos
            elif 'ready' in name:
                ready_pose = pos

        if object_name is None or drawer_name is None:
            print("[Error] Could not find required object or drawer in the environment.")
            return

        # Assume robot starts at some initial location
        robot_location = None
        for name, pos in positions.items():
            if 'robot' in name:
                robot_location = pos
                break

        # === Step 1: Move to object location if not already there ===
        try:
            if robot_location is not None and robot_location != object_pos:
                print(f"[Task] Moving robot from {robot_location} to object at {object_pos}")
                obs, reward, done = execute_go(env, task, robot_location, object_pos)
                robot_location = object_pos
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")
            return

        # === Step 2: Pick object from floor (sets need-ready) ===
        try:
            print(f"[Task] Picking up {object_name} at {object_pos}")
            obs, reward, done = execute_pick(env, task, object_name, object_pos)
            # After this, (need-ready) is set and must be cleared before next action
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # === Step 3: Go to ready pose to clear (need-ready) ===
        try:
            print(f"[Task] Going to ready pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(env, task, object_pos)
            robot_location = ready_pose if ready_pose is not None else object_pos  # Update location
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # === Step 4: Move to drawer location ===
        try:
            if robot_location != drawer_pos:
                print(f"[Task] Moving robot from {robot_location} to drawer at {drawer_pos}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_pos)
                robot_location = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # === Step 5: Place object into drawer (sets need-ready) ===
        try:
            print(f"[Task] Placing {object_name} into {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_place(env, task, object_name, drawer_name, drawer_pos)
            # After this, (need-ready) is set and must be cleared before next action
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")
            return

        # === Step 6: Go to ready pose to clear (need-ready) again ===
        try:
            print(f"[Task] Going to ready pose to clear (need-ready) after placing")
            obs, reward, done = execute_go_ready(env, task, drawer_pos)
            robot_location = ready_pose if ready_pose is not None else drawer_pos
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        # === Task Complete ===
        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
