# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that may block further actions.
        # We must ensure that after actions that set (need-ready), we call the appropriate skill to clear it.
        # In the provided domain, execute_go_ready clears (need-ready).
        # So, after any action that sets (need-ready), we must call execute_go_ready before proceeding.

        # === Example Plan Execution ===
        # For demonstration, we simulate a generic plan with exploration for (need-ready).
        # In a real scenario, you would parse the oracle plan and execute each step accordingly.

        # Let's assume a plan: pick an object from the floor, place it in a drawer, and close the drawer.
        # We'll use only the available skills and handle (need-ready) as per feedback.

        # Example object and location names (replace with actual names from your environment)
        # For demonstration, we use the first object and drawer found in positions
        object_name = None
        drawer_name = None
        handle_name = None
        robot_location = None
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
            elif 'handle' in name and handle_name is None:
                handle_name = name
            elif 'object' in name and object_name is None:
                object_name = name
            elif 'robot' in name and robot_location is None:
                robot_location = name
        # Fallbacks if not found
        if object_name is None:
            object_name = 'object_1'
        if drawer_name is None:
            drawer_name = 'drawer_1'
        if handle_name is None:
            handle_name = 'handle_1'
        if robot_location is None:
            robot_location = 'ready-pose'

        # Assume locations
        floor_location = 'floor'
        drawer_location = 'drawer_area'
        bin_location = 'bin'
        ready_pose = 'ready-pose'

        # Track if need-ready is set
        need_ready = False

        # 1. Move to object location (floor)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=floor_location
            )
        except Exception as e:
            print(f"[Error] execute_go to floor: {e}")

        # 2. Pick object from floor (sets need-ready)
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_name,
                location=floor_location
            )
            need_ready = True
        except Exception as e:
            print(f"[Error] execute_pick: {e}")

        # 3. If need-ready, go to ready pose to clear it
        if need_ready:
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=floor_location
                )
                need_ready = False
            except Exception as e:
                print(f"[Error] execute_go_ready: {e}")

        # 4. Move to drawer location
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=drawer_location
            )
        except Exception as e:
            print(f"[Error] execute_go to drawer: {e}")

        # 5. Place object into drawer (sets need-ready)
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_name,
                drawer_name=drawer_name,
                location=drawer_location
            )
            need_ready = True
        except Exception as e:
            print(f"[Error] execute_place: {e}")

        # 6. If need-ready, go to ready pose to clear it
        if need_ready:
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=drawer_location
                )
                need_ready = False
            except Exception as e:
                print(f"[Error] execute_go_ready: {e}")

        # 7. Push drawer closed
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location
            )
        except Exception as e:
            print(f"[Error] execute_push: {e}")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
