# run_skeleton_task.py (COMPLETED EXECUTABLE CODE)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # You can use print(positions) to debug mapped keys

        # === EXPLORATION PHASE ===
        # Based on feedback, the problematic predicate is (need-ready).
        # Exploration: correlate robot actions to state to find when (need-ready) is set/unset.
        # We perform exploratory actions to check effects and identify the impact of (need-ready).
        # This will, for example, try a sequence using only available skills.

        # For this code's purpose, we will explore by:
        # 1. Attempting a skill (e.g., execute_pick or execute_go) and checking if task blocks or fails,
        #    indicating a missing precondition such as (need-ready).
        # 2. Attempting to clear the (need-ready) state using the available skills (e.g., execute_go_ready).

        # Extract key objects/locations for demonstration
        try:
            # Try to extract key objects from available positions
            # The names in 'positions' will depend on the scenario.
            # We'll guess some key names for generic illustration:
            robot_loc = None
            for k in positions:
                # Assuming position dict contains robot position
                if 'robot' in k or 'robot_base' in k:
                    robot_loc = k
                    break

            # Find an object and a location to try skills
            trial_object = None
            trial_location = None
            for name in positions:
                if 'obj' in name or 'object' in name or 'ball' in name or 'cube' in name:
                    trial_object = name
                    break
            for name in positions:
                if 'drawer' in name and 'handle' not in name:
                    trial_location = name
                    break

            # Set up generic names if auto-detection fails
            if trial_object is None:
                # If no generic object found, pick any
                trial_object = list(positions.keys())[0]
            if trial_location is None:
                trial_location = list(positions.keys())[0]  # Might be improved on real mappings

            print(f"[Exploration] Using object: {trial_object} at location: {trial_location}")
        except Exception as e:
            print(f"[Exploration] Could not auto-extract objects/locations: {e}")
            trial_object = None
            trial_location = None

        # 1. Try executing a skill that is likely to encounter the (need-ready) issue
        exploration_success = False
        if trial_object and trial_location:
            try:
                print("[Exploration] Attempting to execute a skill that might set (need-ready) ...")
                # Try execute_pick (if exists), otherwise try another available skill
                if 'execute_pick' in globals():
                    obs, reward, done = execute_pick(env, task, trial_object, trial_location)
                    print("[Exploration] execute_pick completed.")
                elif 'execute_pick_object' in globals():
                    obs, reward, done = execute_pick_object(env, task, trial_object, trial_location)
                    print("[Exploration] execute_pick_object completed.")
                else:
                    # Try any other available skill as fallback
                    obs, reward, done = execute_sweep(env, task, trial_object, trial_location)
                    print("[Exploration] execute_sweep completed.")
                exploration_success = True
            except Exception as e:
                print(f"[Exploration] Skill execution failed (possibly blocked by missing predicate): {e}")

        # 2. (need-ready) EXPLORATION: If the last action left the robot in a state where most skills are blocked,
        #    attempt to use execute_go_ready to clear (need-ready)
        #    This directly explores the relationship of the need-ready predicate.

        if 'execute_go' in globals() and trial_location:
            try:
                print("[Exploration] Attempting execute_go (may be blocked by need-ready)...")
                obs, reward, done = execute_go(env, task, trial_location, 'ready-pose')
                print("[Exploration] execute_go executed.")
            except Exception as e:
                print(f"[Exploration] execute_go failed (possibly due to need-ready): {e}")

        # 3. Try using execute_go_ready if (need-ready) is True (i.e., previous actions result in blocking)
        if 'execute_go_ready' in globals() and trial_location:
            try:
                print("[Exploration] Attempting execute_go_ready to clear need-ready...")
                obs, reward, done = execute_go_ready(env, task, trial_location)
                print("[Exploration] execute_go_ready executed.")
            except Exception as e:
                print(f"[Exploration] execute_go_ready failed: {e}")

        # 4. After clearing need-ready, try another skill to verify state transition
        if trial_object and trial_location:
            try:
                print("[Exploration] Attempting another execute_pick to check if need-ready was cleared...")
                if 'execute_pick' in globals():
                    obs, reward, done = execute_pick(env, task, trial_object, trial_location)
                    print("[Exploration] execute_pick after go_ready completed.")
                elif 'execute_pick_object' in globals():
                    obs, reward, done = execute_pick_object(env, task, trial_object, trial_location)
                    print("[Exploration] execute_pick_object after go_ready completed.")
            except Exception as e:
                print(f"[Exploration] execute_pick after go_ready failed: {e}")

        # Summary of exploration
        print("[Exploration] If execute_go_ready successfully allowed further skills to proceed,")
        print("then the missing predicate is (need-ready) which must be cleared after certain actions before proceeding.")

        # === TASK/PLAN EXECUTION (Insert oracle plan logic here as needed) ===
        # For this exploration version, we do not have a specific oracle plan, so we focus on enabling
        # exploration to surface the need-ready predicate issue.

        # If you have an oracle plan, you can iterate through the actions here
        # and insert execute_go_ready as needed when blocked by need-ready.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
