# run_skeleton_task.py (Completed for Predicate Exploration and Skill Integration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills; don't redefine or change this import

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with predicate exploration phase.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Diagnose Missing Predicate ===
        print("[Exploration] Beginning exploration phase to diagnose the 'need-ready' predicate handling.")

        # Try to run a no-op skill and see if (need-ready) blocks other actions.
        # The effect of 'need-ready' in the domain is that most actions require (not (need-ready)) in their precondition
        # Solution: After each skill that sets (need-ready), immediately execute 'execute_go_ready'
        # Approach:
        #  - Try a loop; after each major pick/place, attempt to call execute_go_ready if available

        # Let's get a list of (example) objects and drawers for a generically applicable demo.
        all_objects = [k for k in positions.keys() if "obj" in k or "object" in k]
        all_drawers = [k for k in positions.keys() if "drawer" in k]
        all_locations = [k for k in positions.keys() if "loc" in k or "pose" in k or "room" in k]
        # Fallbacks if names aren't standard; may be adapted to your env

        # STARTING LOCATION
        # (Assume initial robot position is stored in obs somewhere, else default to the first location found)
        robot_current_location = None
        if 'robot' in positions:
            robot_current_location = positions['robot']
        elif len(all_locations) > 0:
            robot_current_location = positions[all_locations[0]]
        else:
            robot_current_location = None

        # DEMONSTRATION PLAN:
        # (1) Go to an object's location, (2) pick object, (3) go ready, (4) place object (if possible), (5) go ready

        # For demonstration, pick the first available object and first available location/drawer.
        picked_object = None
        if len(all_objects) > 0:
            picked_object_name = all_objects[0]
            picked_object_pos = positions[picked_object_name]
        else:
            print("[Exploration] No objects found in positions dictionary.")
            picked_object_name, picked_object_pos = None, None

        # Location to pick from (if available)
        if picked_object_pos is not None and robot_current_location is not None:
            # If not already at target, move to object
            if not np.allclose(robot_current_location, picked_object_pos, atol=0.05):
                print(f"[Skill] Moving robot from {robot_current_location} to object {picked_object_name} at {picked_object_pos}")
                try:
                    obs, reward, done = execute_go(env, task, from_pos=robot_current_location, to_pos=picked_object_pos)
                    robot_current_location = picked_object_pos
                except Exception as e:
                    print(f"[Error] Failed to move: {e}")

            # Try PICK
            print(f"[Skill] Attempting to pick object: {picked_object_name}")
            try:
                obs, reward, done = execute_pick(env, task, target_pos=picked_object_pos)
                picked_object = picked_object_name
            except Exception as e:
                print(f"[Error] Failed to pick object: {e}")

            # After pick, due to (need-ready) predicate, we must call execute_go_ready
            print("[Skill] Executing go_ready to clear (need-ready) predicate.")
            try:
                obs, reward, done = execute_go_ready(env, task, from_pos=picked_object_pos)
                robot_current_location = positions.get('ready-pose', picked_object_pos)
            except Exception as e:
                print(f"[Error] Failed to execute go_ready: {e}")

            # Try placing object in bin or drawer if info available (dummy logic: use first drawer or bin found)
            place_done = False
            target_drawer_or_bin_name = None
            target_drawer_or_bin_loc = None
            if 'bin' in positions:
                target_drawer_or_bin_name = 'bin'
                target_drawer_or_bin_loc = positions['bin']
            elif len(all_drawers) > 0:
                target_drawer_or_bin_name = all_drawers[0]
                target_drawer_or_bin_loc = positions[target_drawer_or_bin_name]

            if picked_object and target_drawer_or_bin_loc is not None:
                # Move to placement location if needed
                if not np.allclose(robot_current_location, target_drawer_or_bin_loc, atol=0.05):
                    print(f"[Skill] Moving to place location {target_drawer_or_bin_name} at {target_drawer_or_bin_loc}")
                    try:
                        obs, reward, done = execute_go(env, task, from_pos=robot_current_location, to_pos=target_drawer_or_bin_loc)
                        robot_current_location = target_drawer_or_bin_loc
                    except Exception as e:
                        print(f"[Error] Failed to move to place location: {e}")

                # Try place to bin or drawer
                print(f"[Skill] Attempting to place {picked_object} to {target_drawer_or_bin_name}")
                try:
                    if target_drawer_or_bin_name == 'bin':
                        obs, reward, done = execute_place(env, task, target_pos=target_drawer_or_bin_loc)
                    else:
                        obs, reward, done = execute_place(env, task, target_pos=target_drawer_or_bin_loc, drawer_name=target_drawer_or_bin_name)
                    place_done = True
                except Exception as e:
                    print(f"[Error] Failed to place object: {e}")

                # Again, 'need-ready' becomes true after place, so clear it
                print("[Skill] Executing go_ready to clear (need-ready) after place.")
                try:
                    obs, reward, done = execute_go_ready(env, task, from_pos=target_drawer_or_bin_loc)
                    robot_current_location = positions.get('ready-pose', target_drawer_or_bin_loc)
                except Exception as e:
                    print(f"[Error] Failed to execute go_ready after place: {e}")

        print("[Exploration] Exploration phase completed.")

        # === Continue with further plan steps or finish
        # If you need additional interaction, build here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
