# run_skeleton_task.py (Completed for Robust Oracle Plan Execution with Safe Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        positions = get_object_positions()

        # --- Start robust exploration and task execution ---
        # List of all objects and drawers in the environment
        object_names = list(positions.keys())
        # This dictionary defines known drawers (you may update it based on your scenario)
        drawer_names = [name for name in object_names if 'drawer' in name and 'handle' not in name]
        handle_names = [name for name in object_names if 'handle' in name]
        dice_names = [name for name in object_names if 'dice' in name or 'cube' in name or 'die' in name]
        bin_names = [name for name in object_names if 'bin' in name or 'trash' in name]

        # Safety: Ensure list is not empty before indexing
        if not dice_names:
            print("[Error] No dice objects found in current object positions.")
            return

        # Pick the first available dice
        dice_obj = dice_names[0]
        dice_pos = positions[dice_obj] if dice_obj in positions else None

        # Find a suitable open drawer and its handle
        chosen_drawer = None
        chosen_handle = None
        for dn in drawer_names:
            # Try to find matching handle for this drawer (by naming convention)
            for hn in handle_names:
                if dn.replace('drawer', '') in hn:
                    chosen_drawer = dn
                    chosen_handle = hn
                    break
            if chosen_drawer and chosen_handle:
                break

        if not chosen_drawer:
            print("[Error] No suitable drawer found with handle.")
            return
        if not chosen_handle:
            print("[Error] No suitable handle found for drawer:", chosen_drawer)
            return

        # --- Check for object existence (from feedback) ---
        for name in [dice_obj, chosen_drawer, chosen_handle]:
            if name not in positions:
                print(f"[Safety Error] Object '{name}' not found in object positions. Aborting.")
                return

        # --- Try performing the task step-by-step with error checks ---
        try:
            # Step 1: Move robot to initial ready pose or close to dice position
            robot_loc = 'ready-pose'
            if 'ready-pose' in positions:
                robot_loc = 'ready-pose'
            else:
                # fallback: use dice position as current location
                robot_loc = dice_pos

            # Step 2: Approach the dice location if needed
            if hasattr(env, 'robot_location') and env.robot_location != dice_pos:
                obs, reward, done = execute_go(env, task, robot_loc, dice_pos)
                robot_loc = dice_pos

            # Step 3: Pick up the dice
            # --- Safety check for predicate (e.g., dice is on floor and robot hand is empty)
            obs, reward, done = execute_pick(env, task, dice_obj, robot_loc)
            if not obs:  # obs == None signals problem
                print("[Task] Failed to pick up dice:", dice_obj)
                return
            print("[Task] Picked up dice:", dice_obj)

            # Step 4: Go to the target drawer location
            drawer_pos = positions[chosen_drawer]
            if robot_loc != drawer_pos:
                obs, reward, done = execute_go(env, task, robot_loc, drawer_pos)
                robot_loc = drawer_pos

            # Step 5: Check if the drawer is unlocked and closed
            # NOTE: In practice, check drawer states if observations allow
            # Here, we use exploration: If drawer-unlocked predicate is not observed, try pulling to test lock
            obs, reward, done = execute_pull(env, task, chosen_drawer, chosen_handle, robot_loc)
            print("[Task] Pulled open drawer:", chosen_drawer)

            # Step 6: Place dice into drawer (safely: only if drawer is open)
            # -- Must ensure drawer is open via predicate or via previous step
            obs, reward, done = execute_place(env, task, dice_obj, chosen_drawer, robot_loc)
            if not obs:
                print(f"[Task] Cannot place {dice_obj} into {chosen_drawer}; placement failed or precondition not met.")
            else:
                print(f"[Task] Placed {dice_obj} into {chosen_drawer}.")

            # Step 7: Optionally, push the drawer to close it
            obs, reward, done = execute_push(env, task, chosen_drawer, robot_loc)
            print(f"[Task] Pushed drawer {chosen_drawer} closed.")

        except Exception as e:
            print("[Error during skill execution]:", e)
            return

        # --- Exploration for missing predicates (See feedback & exploration knowledge) ---
        # E.g., for unknown predicates (identified, temperature-known, lock-known, etc.)
        # Safely check which are not present in obs; perform the corresponding exploration actions

        missing_predicates = []
        observation_predicates = set()
        # Assuming obs contains a dict of predicates
        if isinstance(obs, dict):
            observation_predicates = set(obs.keys())
        # For this task, let's focus on 'lock-known'
        if 'lock-known' not in observation_predicates:
            print("[Exploration] Predicate 'lock-known' missing, performing safe exploration pull to reveal lock status.")
            try:
                obs, reward, done = execute_pull(env, task, chosen_drawer, chosen_handle, robot_loc)
                print("[Exploration] Executed 'pull' for lock-known exploration.")
            except Exception as ex:
                print("[Exploration Error]:", ex)

        # --- End plan ---
        print("[Task] Task execution completed.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()