# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer_top) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer_top) is missing.
        # We need to explore the environment to determine the lock state of the drawer.
        # We will use the available skills to do this.

        # Assumptions:
        # - There is a drawer named 'drawer_top' in the environment.
        # - The robot starts at some initial location.
        # - The handle for the drawer is on the floor and must be picked up.
        # - The robot must move to the handle, pick it up, move to the drawer, and attempt to pull the drawer.
        # - If the drawer is locked, the pull will fail or indicate the lock state.

        # Step 1: Get relevant object and location names
        try:
            drawer_name = 'drawer_top'
            handle_name = None
            for obj in positions:
                if 'handle' in obj and 'drawer_top' in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                # Fallback: pick any handle
                for obj in positions:
                    if 'handle' in obj:
                        handle_name = obj
                        break
            if handle_name is None:
                raise Exception("No handle found for drawer_top in object positions.")

            # Get locations
            handle_pos = positions[handle_name]
            drawer_pos = positions[drawer_name]
            # For simplicity, assume location names are keys in positions
            # If not, use the position tuples for navigation

            # Step 2: Move to handle location
            print(f"[Exploration] Moving to handle {handle_name} at {handle_pos}")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=None,  # Let skill handle current location
                    to_location=handle_name,
                    max_steps=100
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_go to handle: {e}")

            # Step 3: Pick up the handle
            print(f"[Exploration] Picking up handle {handle_name}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=handle_name,
                    location_name=handle_name,
                    max_steps=100
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_pick handle: {e}")

            # Step 4: Move to drawer location
            print(f"[Exploration] Moving to drawer {drawer_name} at {drawer_pos}")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=handle_name,
                    to_location=drawer_name,
                    max_steps=100
                )
            except Exception as e:
                print(f"[Exploration] Error during execute_go to drawer: {e}")

            # Step 5: Attempt to pull the drawer (this will help us determine lock state)
            print(f"[Exploration] Attempting to pull drawer {drawer_name} with handle {handle_name}")
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=handle_name,
                    location_name=drawer_name,
                    max_steps=100
                )
                print("[Exploration] Pull action executed. If drawer was locked, this should reveal lock state.")
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")

            # At this point, the environment or observation should reflect the lock state.
            # The missing predicate (drawer-unlocked drawer_top) can be inferred from the result.
            # You may inspect obs or task state here if needed.

        except Exception as e:
            print(f"[Exploration] Error in exploration phase: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after exploration if needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
