# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the state of the drawer to determine if it is unlocked.
        # We use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assumptions for this code:
        # - There is a drawer named 'drawer1' and a handle object (e.g., 'handle1') attached to it.
        # - The robot starts at some initial location (e.g., 'ready-pose').
        # - The handle is on the floor and must be picked up before pulling the drawer.
        # - The robot must move to the drawer location to interact with it.
        # - The plan is to pick up the handle, move to the drawer, and attempt to pull the drawer to check if it is unlocked.

        # You may need to adjust object/location names based on your environment.
        # For demonstration, we use 'drawer1', 'handle1', and 'drawer_location'.

        # === Step 1: Get relevant object and location names ===
        # These should be present in the positions dictionary or can be inferred from the environment.
        try:
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            # Find the location of the handle (assume it's on the floor at 'handle_location')
            handle_location = positions.get(handle_name + '_location', None)
            drawer_location = positions.get(drawer_name + '_location', None)
            robot_location = positions.get('robot_location', None)
            if handle_location is None or drawer_location is None or robot_location is None:
                # Fallback: try to get any available location
                handle_location = positions.get('handle_location', None)
                drawer_location = positions.get('drawer_location', None)
                robot_location = positions.get('ready-pose', None)
        except Exception as e:
            print("[Error] Could not determine object or location names:", e)
            return

        # === Step 2: Move to handle location if not already there ===
        try:
            if robot_location != handle_location and handle_location is not None and robot_location is not None:
                print(f"[Exploration] Moving robot from {robot_location} to handle at {handle_location}")
                obs, reward, done = execute_go(env, task, robot_location, handle_location)
                robot_location = handle_location
        except Exception as e:
            print("[Error] Failed to move to handle location:", e)
            return

        # === Step 3: Pick up the handle ===
        try:
            print(f"[Exploration] Picking up handle: {handle_name} at {handle_location}")
            obs, reward, done = execute_pick(env, task, handle_name, handle_location)
        except Exception as e:
            print("[Error] Failed to pick up handle:", e)
            return

        # === Step 4: Move to drawer location ===
        try:
            if robot_location != drawer_location and drawer_location is not None and robot_location is not None:
                print(f"[Exploration] Moving robot from {robot_location} to drawer at {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print("[Error] Failed to move to drawer location:", e)
            return

        # === Step 5: Attempt to pull the drawer to check lock status ===
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name} at {drawer_location}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            # If the action succeeds, the drawer is unlocked.
            print("[Exploration] Drawer pull succeeded: (drawer-unlocked drawer1) is TRUE")
        except Exception as e:
            # If the action fails, the drawer may be locked.
            print("[Exploration] Drawer pull failed: (drawer-unlocked drawer1) is FALSE or action precondition not met")
            print("[Error] Pull action exception:", e)
            # Optionally, handle further exploration or report the missing predicate.

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan or Task Logic ===
        # At this point, you can proceed with the main task plan, now knowing the lock status.
        # For demonstration, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()