# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the "execute_go_ready" skill to clear it.

        # For demonstration, let's assume a generic oracle plan:
        # 1. Move to object location
        # 2. Pick object from floor
        # 3. Go to drawer
        # 4. Place object into drawer
        # (Each pick/place sets (need-ready), so we must call execute_go_ready after each.)

        # For this example, let's assume the following object and locations:
        # (You should replace these with actual names from your environment/observation)
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        ready_pose = 'ready-pose'
        robot_current_location = None

        # Try to infer object and drawer from positions dict
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif 'object' in name or 'obj' in name:
                object_name = name
                object_pos = pos
            elif 'robot' in name and robot_current_location is None:
                robot_current_location = name

        # Fallbacks if not found
        if robot_current_location is None:
            robot_current_location = 'start'  # or some default

        # === Example Plan Execution ===
        try:
            # 1. Move to object location (if not already there)
            if robot_current_location != object_pos:
                print(f"[Task] Moving robot from {robot_current_location} to object at {object_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=object_pos
                )
                robot_current_location = object_pos
                if done:
                    print("[Task] Task ended during move to object!")
                    return

            # 2. Pick object from floor
            print(f"[Task] Picking up object {object_name} at {object_pos}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name,
                object_pos
            )
            if done:
                print("[Task] Task ended during pick!")
                return

            # After pick, (need-ready) is set, so we must go to ready-pose
            print("[Task] Going to ready-pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_pos
            )
            robot_current_location = ready_pose
            if done:
                print("[Task] Task ended during go_ready!")
                return

            # 3. Move to drawer location
            if robot_current_location != drawer_pos:
                print(f"[Task] Moving robot from {robot_current_location} to drawer at {drawer_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=drawer_pos
                )
                robot_current_location = drawer_pos
                if done:
                    print("[Task] Task ended during move to drawer!")
                    return

            # 4. Place object into drawer
            print(f"[Task] Placing object {object_name} into drawer {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name,
                drawer_name,
                drawer_pos
            )
            if done:
                print("[Task] Task ended during place!")
                return

            # After place, (need-ready) is set, so we must go to ready-pose
            print("[Task] Going to ready-pose to clear (need-ready) after place")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos
            )
            robot_current_location = ready_pose
            if done:
                print("[Task] Task ended after go_ready!")
                return

            print("[Task] Plan executed successfully.")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()

# Notes:
# - This code demonstrates the exploration and handling of the (need-ready) predicate as required by the domain and feedback.
# - The plan is generic; you should adapt object/drawer names and positions to your actual environment.
# - Only predefined skills are used, and after any action that sets (need-ready), execute_go_ready is called to clear it.
# - All imports and structure from the skeleton are preserved.