# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # === Example Plan Execution ===
        # For demonstration, we will:
        # 1. Pick up an object from the floor (execute_pick)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Place the object into a drawer (execute_place)
        # 4. Go to ready pose (execute_go_ready) again if needed

        # For this example, we will use the first object and first location found in positions
        # You should adapt this logic to your actual oracle plan and object names

        # Find an object on the floor and a drawer
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        robot_pos = None

        # Identify objects and locations
        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif 'object' in name and object_name is None:
                object_name = name
                object_pos = pos
            elif 'robot' in name:
                robot_pos = pos

        if object_name is None or drawer_name is None or robot_pos is None:
            print("[Error] Could not find required objects or robot in the environment.")
            return

        # 1. Move robot to the object's location if not already there
        try:
            if positions['robot'] != object_pos:
                print(f"[Task] Moving robot to object location: {object_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions['robot'],
                    to_location=object_pos
                )
                positions['robot'] = object_pos
        except Exception as e:
            print(f"[Error] Failed to move robot to object: {e}")
            return

        # 2. Pick up the object from the floor
        try:
            print(f"[Task] Picking up object: {object_name} at {object_pos}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_name,
                location=object_pos
            )
        except Exception as e:
            print(f"[Error] Failed to pick up object: {e}")
            return

        # 3. After pick, (need-ready) is set. Must call execute_go_ready to clear it.
        try:
            print("[Task] Going to ready pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_pos
            )
            positions['robot'] = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # 4. Move robot to the drawer location
        try:
            if positions['robot'] != drawer_pos:
                print(f"[Task] Moving robot to drawer location: {drawer_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions['robot'],
                    to_location=drawer_pos
                )
                positions['robot'] = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move robot to drawer: {e}")
            return

        # 5. Place the object into the drawer
        try:
            print(f"[Task] Placing object: {object_name} into drawer: {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_name,
                drawer_name=drawer_name,
                location=drawer_pos
            )
        except Exception as e:
            print(f"[Error] Failed to place object into drawer: {e}")
            return

        # 6. After place, (need-ready) is set again. Call execute_go_ready.
        try:
            print("[Task] Going to ready pose to clear (need-ready) after placing object")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos
            )
            positions['robot'] = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
