# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to identify missing predicates.
        # Since we cannot define new skills, we use the available ones to interact and observe.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore the environment using available skills
        # For demonstration, we attempt to use each skill in a safe, generic way
        # and print out any exceptions or observations that may indicate missing predicates.

        print("=== Exploration Phase: Trying available skills to identify missing predicates ===")
        exploration_results = {}
        for skill_name in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                # For each skill, try to call it with plausible arguments from positions
                # Since we don't know the exact arguments, we try to infer them
                if skill_name == 'execute_go':
                    # Try moving between two locations if available
                    locs = [k for k, v in positions.items() if 'location' in k or 'pose' in k or 'drawer' in k]
                    if len(locs) >= 2:
                        from_loc = locs[0]
                        to_loc = locs[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_pick':
                    # Try picking an object from a location
                    objs = [k for k in positions if 'object' in k or 'obj' in k or 'handle' in k]
                    locs = [k for k in positions if 'location' in k or 'pose' in k or 'drawer' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_place':
                    # Try placing an object into a location
                    objs = [k for k in positions if 'object' in k or 'obj' in k]
                    locs = [k for k in positions if 'drawer' in k or 'bin' in k or 'location' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        obs, reward, done = execute_place(env, task, obj, loc)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_push':
                    # Try pushing a drawer if available
                    drawers = [k for k in positions if 'drawer' in k]
                    if drawers:
                        drawer = drawers[0]
                        obs, reward, done = execute_push(env, task, drawer)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_pull':
                    # Try pulling a handle if available
                    handles = [k for k in positions if 'handle' in k]
                    drawers = [k for k in positions if 'drawer' in k]
                    if handles and drawers:
                        handle = handles[0]
                        drawer = drawers[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_sweep':
                    # Try sweeping an object
                    objs = [k for k in positions if 'object' in k or 'obj' in k]
                    locs = [k for k in positions if 'location' in k or 'pose' in k]
                    if objs and locs:
                        obj = objs[0]
                        loc = locs[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_rotate':
                    # Try rotating an object if possible
                    objs = [k for k in positions if 'object' in k or 'obj' in k]
                    if objs:
                        obj = objs[0]
                        obs, reward, done = execute_rotate(env, task, obj)
                        exploration_results[skill_name] = (obs, reward, done)
                elif skill_name == 'execute_gripper':
                    # Try gripper action (no-op)
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill_name] = (obs, reward, done)
                print(f"[Exploration] Skill {skill_name} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill_name} failed: {e}")
                exploration_results[skill_name] = str(e)

        print("=== Exploration Results ===")
        for skill, result in exploration_results.items():
            print(f"Skill: {skill}, Result: {result}")

        print("=== End of Exploration Phase ===")

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan step-by-step.
        # For this template, you would insert the plan steps here, using only the available skills.
        # Example (commented out, as actual plan is not provided):
        #
        # obs, reward, done = execute_go(env, task, 'ready-pose', 'drawer_location')
        # obs, reward, done = execute_pull(env, task, 'drawer1', 'handle1')
        # obs, reward, done = execute_pick(env, task, 'object1', 'drawer1')
        # obs, reward, done = execute_place(env, task, 'object1', 'bin')
        #
        # if done:
        #     print("[Task] Task completed successfully!")
        #     return

        # TODO: Insert actual oracle plan steps here using only the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()