# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # The feedback suggests a missing predicate is blocking planning.
        # We'll attempt to use available skills to explore the environment and infer which predicate is missing.
        # Since we cannot define new skills, we use the available ones to interact and observe.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers to see if any action fails due to a missing predicate
        # We'll print out the results for each action to help identify the missing predicate

        # Get all objects and locations from positions
        object_names = [name for name in positions.keys() if not name.startswith('drawer') and not name.startswith('bin')]
        drawer_names = [name for name in positions.keys() if name.startswith('drawer')]
        location_names = list(set([v for v in positions.values()]))

        # For demonstration, we assume the robot starts at a known location
        # We'll try to move to each location and attempt to pick/place/pull/push as appropriate

        # 1. Try to move to each location
        for loc in location_names:
            try:
                print(f"[Exploration] Trying to move to location: {loc}")
                # We need a 'from' and 'to' location; assume robot starts at 'ready-pose'
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='ready-pose',
                    to_location=loc
                )
                if done:
                    print(f"[Exploration] Moved to {loc} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # 2. Try to pick each object from the floor
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to pick object: {obj}")
                # Assume all objects are on the floor at their position
                obj_pos = positions[obj]
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj,
                    location=obj_pos
                )
                if done:
                    print(f"[Exploration] Picked {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj}: {e}")

        # 3. Try to open each drawer (pull)
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to pull drawer: {drawer}")
                # For pulling, we need a handle object and location
                # Try to find a handle for this drawer
                handle_obj = None
                for obj in object_names:
                    if 'handle' in obj and drawer in obj:
                        handle_obj = obj
                        break
                if handle_obj is not None:
                    handle_pos = positions[handle_obj]
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer,
                        handle_name=handle_obj,
                        location=handle_pos
                    )
                    if done:
                        print(f"[Exploration] Pulled {drawer} open successfully.")
                else:
                    print(f"[Exploration] No handle found for {drawer}.")
            except Exception as e:
                print(f"[Exploration] Failed to pull {drawer}: {e}")

        # 4. Try to push each drawer closed
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to push drawer: {drawer}")
                # Assume robot is at the drawer's location
                drawer_pos = positions[drawer]
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer,
                    location=drawer_pos
                )
                if done:
                    print(f"[Exploration] Pushed {drawer} closed successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to push {drawer}: {e}")

        # 5. Try to place each object into each drawer
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Trying to place {obj} into {drawer}")
                    drawer_pos = positions[drawer]
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=obj,
                        drawer_name=drawer,
                        location=drawer_pos
                    )
                    if done:
                        print(f"[Exploration] Placed {obj} into {drawer} successfully.")
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} into {drawer}: {e}")

        # 6. Try to sweep each object
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to sweep object: {obj}")
                obj_pos = positions[obj]
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    object_name=obj,
                    location=obj_pos
                )
                if done:
                    print(f"[Exploration] Swept {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # 7. Try to rotate each object (if applicable)
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to rotate object: {obj}")
                obj_pos = positions[obj]
                obs, reward, done = execute_rotate(
                    env,
                    task,
                    object_name=obj,
                    location=obj_pos
                )
                if done:
                    print(f"[Exploration] Rotated {obj} successfully.")
            except Exception as e:
                print(f"[Exploration] Failed to rotate {obj}: {e}")

        # 8. Try to use the gripper action (no-op)
        try:
            print(f"[Exploration] Trying to use gripper action")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Gripper action executed successfully.")
        except Exception as e:
            print(f"[Exploration] Failed to execute gripper action: {e}")

        print("[Exploration] Exploration phase complete. Review the above logs for any failed actions and missing predicates.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate from the exploration phase,
        # update the domain/problem or inform the user to add the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
