# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding handle_top)
        # The plan failed because a predicate is missing (likely lock-known for the handle/drawer).
        # We need to perform an exploration action to discover the lock state before proceeding.

        # For this example, let's assume:
        # - The robot starts at 'ready-pose'
        # - The handle is named 'handle_top'
        # - The drawer is named 'drawer_top'
        # - The location of the handle is 'drawer_top_front' (or similar)
        # - The robot must pick up the handle, then execute_pull to discover lock-known

        # You may need to adjust object/location names to match your environment.

        # Step 1: Move to the handle's location
        try:
            handle_name = 'handle_top'
            drawer_name = 'drawer_top'
            handle_location = positions.get(handle_name + '_location', None)
            if handle_location is None:
                # Fallback: try to get the drawer's front location
                handle_location = positions.get(drawer_name + '_front', None)
            if handle_location is None:
                # Fallback: use a default location
                handle_location = positions.get('default_handle_location', None)
            if handle_location is None:
                raise Exception("Handle location not found in positions dictionary.")

            # Move robot to handle location (assume robot starts at 'ready-pose')
            print(f"[Exploration] Moving to handle location: {handle_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=handle_location
            )
            if done:
                print("[Exploration] Task ended unexpectedly during move to handle.")
                return

        except Exception as e:
            print(f"[Exploration] Error during move to handle: {e}")
            return

        # Step 2: Pick up the handle
        try:
            print(f"[Exploration] Picking up handle: {handle_name} at {handle_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_location
            )
            if done:
                print("[Exploration] Task ended unexpectedly during pick handle.")
                return
        except Exception as e:
            print(f"[Exploration] Error during pick handle: {e}")
            return

        # Step 3: Execute pull to discover lock-known predicate
        try:
            print(f"[Exploration] Pulling drawer: {drawer_name} with handle: {handle_name} at {handle_location}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=handle_location
            )
            if done:
                print("[Exploration] Task ended unexpectedly during pull.")
                return
        except Exception as e:
            print(f"[Exploration] Error during pull: {e}")
            return

        # At this point, the lock-known predicate should be discovered for the drawer/handle.
        print("[Exploration] Exploration phase complete. Missing predicate (lock-known) should now be known.")

        # === Continue with Oracle Plan (if any) ===
        # Insert further plan steps here as needed, using only predefined skills.
        # For example, you could now open the drawer, pick/place objects, etc.

        # Example: Move back to ready-pose
        try:
            print("[Task] Returning to ready-pose.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=handle_location,
                to_location='ready-pose'
            )
            if done:
                print("[Task] Task ended after returning to ready-pose.")
                return
        except Exception as e:
            print(f"[Task] Error during return to ready-pose: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
