# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer_top) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer_top) is missing.
        # We need to explore the environment to determine the lock state of the drawer.
        # We will use the available skills to do this.

        # Assumptions:
        # - There is a drawer named 'drawer_top' in the environment.
        # - The robot starts at some initial location.
        # - The handle for the drawer is on the floor and can be picked up.
        # - The robot must pick up the handle, move to the drawer, and attempt to pull the drawer to determine if it is unlocked.

        # Step 1: Get relevant object and location names
        # These should be replaced with actual names from the environment if available.
        # For demonstration, we use common names.
        drawer_name = 'drawer_top'
        handle_name = 'handle_top'
        bin_name = 'bin'
        ready_pose = 'ready-pose'

        # Try to get positions for handle and drawer
        try:
            handle_pos = positions[handle_name]
        except KeyError:
            print(f"[Exploration] Handle '{handle_name}' not found in positions. Exploration aborted.")
            return
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            print(f"[Exploration] Drawer '{drawer_name}' not found in positions. Exploration aborted.")
            return

        # Step 2: Move to handle location if not already there
        # Assume we can get the robot's current location from the observation
        try:
            robot_loc = obs['robot_position_name'] if 'robot_position_name' in obs else None
        except Exception:
            robot_loc = None

        if robot_loc is None:
            # Fallback: try to infer from positions or set to a default
            robot_loc = 'ready-pose'

        if robot_loc != handle_pos['location']:
            try:
                print(f"[Exploration] Moving from {robot_loc} to handle at {handle_pos['location']}")
                obs, reward, done = execute_go(env, task, robot_loc, handle_pos['location'])
                robot_loc = handle_pos['location']
            except Exception as e:
                print(f"[Exploration] Error during execute_go to handle: {e}")
                return

        # Step 3: Pick up the handle
        try:
            print(f"[Exploration] Picking up handle '{handle_name}' at {robot_loc}")
            obs, reward, done = execute_pick(env, task, handle_name, robot_loc)
        except Exception as e:
            print(f"[Exploration] Error during execute_pick for handle: {e}")
            return

        # Step 4: Move to drawer location if not already there
        if robot_loc != drawer_pos['location']:
            try:
                print(f"[Exploration] Moving from {robot_loc} to drawer at {drawer_pos['location']}")
                obs, reward, done = execute_go(env, task, robot_loc, drawer_pos['location'])
                robot_loc = drawer_pos['location']
            except Exception as e:
                print(f"[Exploration] Error during execute_go to drawer: {e}")
                return

        # Step 5: Attempt to pull the drawer using the handle
        # This will help us determine if the drawer is locked or unlocked
        try:
            print(f"[Exploration] Attempting to pull drawer '{drawer_name}' with handle '{handle_name}' at {robot_loc}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_loc)
            print("[Exploration] Pull action executed. If drawer opened, it was unlocked.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] Drawer may be locked or another issue occurred.")

        # Step 6: (Optional) Place handle in bin or return to ready pose
        try:
            print(f"[Exploration] Placing handle '{handle_name}' in bin at {robot_loc}")
            obs, reward, done = execute_place(env, task, handle_name, bin_name, robot_loc)
        except Exception as e:
            print(f"[Exploration] Error during execute_place for handle: {e}")

        try:
            print(f"[Exploration] Returning to ready pose from {robot_loc}")
            obs, reward, done = execute_go(env, task, robot_loc, ready_pose)
        except Exception as e:
            print(f"[Exploration] Error during execute_go to ready pose: {e}")

        print("[Exploration] Exploration phase complete. Check logs or state for (drawer-unlocked drawer_top).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
