# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # We will perform an exploration phase to check if the environment requires the robot to be in a "ready" state
        # before or after certain actions, and if so, we will execute the corresponding skill to resolve it.

        # For demonstration, we will attempt to execute a sequence of available skills,
        # and if we encounter a situation where (need-ready) is required, we will call execute_go_ready.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Example: Suppose we want to pick up an object and place it somewhere.
        # We will simulate a generic plan using only the available skills.

        # For this example, let's assume we have at least one object and one location.
        # We'll use the first object and two locations from the positions dictionary.
        object_names = [name for name in positions if not name.startswith('robot') and not name.startswith('drawer')]
        location_names = [name for name in positions if name.startswith('location') or name.startswith('bin') or name.startswith('drawer')]

        # Fallbacks in case names are not as expected
        if len(object_names) == 0:
            print("[Exploration] No objects found in positions. Exiting exploration.")
            return
        if len(location_names) < 2:
            print("[Exploration] Not enough locations found in positions. Exiting exploration.")
            return

        obj = object_names[0]
        from_location = location_names[0]
        to_location = location_names[1]

        # Step 1: Move to the object's location
        print(f"[Exploration] Moving to {from_location} to approach {obj}.")
        try:
            obs, reward, done = execute_go(env, task, from_location, from_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 2: Try to pick the object
        print(f"[Exploration] Attempting to pick {obj} at {from_location}.")
        try:
            obs, reward, done = execute_pick(env, task, obj, from_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            # If error is related to (need-ready), try to resolve it
            if 'need-ready' in str(e) or 'ready' in str(e):
                print("[Exploration] Detected missing (need-ready) predicate. Executing execute_go_ready to resolve.")
                try:
                    obs, reward, done = execute_go_ready(env, task, from_location)
                    # Retry pick after going to ready
                    obs, reward, done = execute_pick(env, task, obj, from_location)
                except Exception as e2:
                    print(f"[Exploration] Error during execute_go_ready or retry pick: {e2}")

        # Step 3: Move to the destination location
        print(f"[Exploration] Moving to {to_location} to place {obj}.")
        try:
            obs, reward, done = execute_go(env, task, from_location, to_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 4: Try to place the object
        print(f"[Exploration] Attempting to place {obj} at {to_location}.")
        try:
            obs, reward, done = execute_place(env, task, obj, to_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_place: {e}")
            # If error is related to (need-ready), try to resolve it
            if 'need-ready' in str(e) or 'ready' in str(e):
                print("[Exploration] Detected missing (need-ready) predicate. Executing execute_go_ready to resolve.")
                try:
                    obs, reward, done = execute_go_ready(env, task, to_location)
                    # Retry place after going to ready
                    obs, reward, done = execute_place(env, task, obj, to_location)
                except Exception as e2:
                    print(f"[Exploration] Error during execute_go_ready or retry place: {e2}")

        # Step 5: Exploration complete
        print("[Exploration] Exploration phase complete. If (need-ready) was missing, it should now be handled.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
