# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding handle_top)
        # The goal is to determine which predicate is missing by using only the available skills.
        # We will attempt to execute the available skills and observe the environment's response.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we will:
        # 1. Try to pick up the handle (e.g., 'handle_top') if it is on the floor and the robot's hand is empty.
        # 2. Try to pull a drawer using the handle, and observe if any predicate (e.g., lock-known) is missing.
        # 3. If an action fails, log the error and continue.

        # The following is a generic exploration loop using only the available skills.

        # Example object and location names (replace with actual names from your environment)
        handle_name = None
        drawer_name = None
        handle_location = None
        robot_location = None

        # Try to find a handle and a drawer from the positions dictionary
        for obj_name, obj_info in positions.items():
            if 'handle' in obj_name:
                handle_name = obj_name
                handle_location = obj_info.get('location', None)
            if 'drawer' in obj_name:
                drawer_name = obj_name
        # Try to get robot's current location
        robot_location = positions.get('robot', {}).get('location', None)

        # If not found, use default names (these should be replaced with actual environment names)
        if handle_name is None:
            handle_name = 'handle_top'
        if drawer_name is None:
            drawer_name = 'drawer_main'
        if handle_location is None:
            handle_location = 'floor'
        if robot_location is None:
            robot_location = 'ready-pose'

        print(f"[Exploration] Handle: {handle_name}, Drawer: {drawer_name}, Handle Location: {handle_location}, Robot Location: {robot_location}")

        # Step 1: Move robot to handle location if not already there
        try:
            if robot_location != handle_location:
                print(f"[Exploration] Moving robot from {robot_location} to {handle_location}")
                obs, reward, done, info = execute_go(env, task, robot_location, handle_location)
                robot_location = handle_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Step 2: Try to pick up the handle
        try:
            print(f"[Exploration] Attempting to pick up handle: {handle_name} at {handle_location}")
            obs, reward, done, info = execute_pick(env, task, handle_name, handle_location)
            # If successful, robot should now be holding the handle
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # Step 3: Move to drawer location if needed
        drawer_location = positions.get(drawer_name, {}).get('location', None)
        if drawer_location is None:
            drawer_location = 'drawer_area'
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done, info = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go to drawer: {e}")

        # Step 4: Try to pull the drawer using the handle
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} with handle: {handle_name} at {drawer_location}")
            obs, reward, done, info = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            # If the action fails due to a missing predicate (e.g., lock-known), this will be revealed in the error or environment feedback.
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] This may indicate a missing predicate such as lock-known or drawer-unlocked.")

        # Step 5: (Optional) Try other skills if needed for further exploration
        # For example, try execute_push, execute_sweep, etc., as needed.

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()