# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the "execute_go_ready" skill to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Pick up an object from the floor (execute_pick)
        # 2. Place it into a drawer (execute_place)
        # 3. If (need-ready) is set, go to ready pose (execute_go_ready)
        # 4. Move to another location (execute_go)
        # 5. Push the drawer closed (execute_push)
        # (The actual plan would be generated by a planner, but here we show the control flow.)

        # For this example, we will use placeholder object and location names.
        # Replace these with actual names from your environment as needed.
        object_name = None
        drawer_name = None
        location_from = None
        location_to = None

        # Try to find a suitable object and drawer from positions
        for name in positions:
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
            elif 'object' in name and object_name is None:
                object_name = name
            elif 'bin' in name and location_to is None:
                location_to = name
            elif location_from is None:
                location_from = name

        # Fallbacks if not found
        if object_name is None:
            object_name = 'object_1'
        if drawer_name is None:
            drawer_name = 'drawer_1'
        if location_from is None:
            location_from = 'location_1'
        if location_to is None:
            location_to = 'location_2'

        # Track if need-ready is set (simulate the predicate)
        need_ready = False

        # 1. Pick up the object from the floor
        try:
            print(f"[Task] Picking up {object_name} at {location_from}")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_name,
                location=location_from
            )
            # According to the domain, execute_pick sets (need-ready)
            need_ready = True
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")

        # 2. If need-ready, go to ready pose
        if need_ready:
            try:
                print("[Task] Going to ready pose to clear need-ready")
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=location_from
                )
                need_ready = False
            except Exception as e:
                print(f"[Error] Failed to go to ready pose: {e}")

        # 3. Place the object into the drawer
        try:
            print(f"[Task] Placing {object_name} into {drawer_name} at {location_to}")
            obs, reward, done = execute_place(
                env,
                task,
                obj=object_name,
                drawer=drawer_name,
                location=location_to
            )
            # execute_place sets (need-ready)
            need_ready = True
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")

        # 4. If need-ready, go to ready pose
        if need_ready:
            try:
                print("[Task] Going to ready pose to clear need-ready")
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=location_to
                )
                need_ready = False
            except Exception as e:
                print(f"[Error] Failed to go to ready pose: {e}")

        # 5. Move to another location (if needed)
        try:
            print(f"[Task] Moving from {location_to} to {location_from}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=location_to,
                to_location=location_from
            )
        except Exception as e:
            print(f"[Error] Failed to move: {e}")

        # 6. Push the drawer closed
        try:
            print(f"[Task] Pushing {drawer_name} closed at {location_from}")
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=location_from
            )
        except Exception as e:
            print(f"[Error] Failed to push drawer: {e}")

        print("[Task] Task execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
